<?php

namespace Drupal\monarch_advanced_entity_usage\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\monarch_advanced_entity_usage\AdvancedEntityUsage;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Returns responses for Monarch Advanced Entity Usage routes.
 */
final class MonarchAdvancedEntityUsageController extends ControllerBase {

  const LIST_MAX = 100;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The advanced_entity_usage service.
   */
  protected AdvancedEntityUsage $advancedEntityUsage;

  /**
   * The entity_type.bundle.info service.
   */
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $instance = parent::create($container);

    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->currentUser = $container->get('current_user');
    $instance->advancedEntityUsage = $container->get('advanced_entity_usage');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');

    return $instance;
  }

  /**
   * Title callback for the entity list page.
   */
  public function entityTypesTitle(): string {
    return $this->t('Advanced Entity Usage: Entity Types');
  }

  /**
   * Title callback for the entity list page.
   */
  public function entityListTitle(string $entity_type_id): string {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    return $this->t('Advanced Entity Usage: @type List', ['@type' => $entity_type->getLabel()]);
  }

    /**
   * Title callback for the usage list page.
   */
  public function usageListTitle(string $entity_type_id, string $entity_id): string {
    $entity = $this->entityTypeManager->getStorage($entity_type_id)->load($entity_id);
    if ($entity) {
      $label = $entity->label() ?: $this->t('@type @id', [
        '@type' => $entity->getEntityType()->getLabel(),
        '@id' => $entity->id(),
      ]);

      return $this->t('Advanced Entity Usage: @label', ['@label' => $label]);
    }

    return $this->t('Advanced Entity Usage');
  }

  /**
   * Checks if an entity type has more than one bundle.
   */
  public function hasBundles(string $entity_type_id): bool {
    $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
    return count($bundles) > 1;
  }

  /**
   * Builds the response.
   */
  public function entityTypes(): array {
    $build = [];

    $definitions = $this->entityTypeManager->getDefinitions();
    $map = $this->advancedEntityUsage->getFieldMap();

    ksort($definitions);

    $build['content'] = [
      '#theme' => 'plain_table',
      '#data' => [],
    ];

    foreach ($definitions as $entity_type_id => $definition) {
      if (isset($map[$entity_type_id]) && $definition instanceof ContentEntityTypeInterface) {
        $link = Link::createFromRoute($definition->getLabel(), 'monarch_advanced_entity_usage.entity_list', ['entity_type_id' => $entity_type_id], ['attributes' => ['target' => '_blank']]);

        if ($link->getUrl()->access($this->currentUser, FALSE)) {
          $build['content']['#data'][$entity_type_id] = [
            'link' => $link->getUrl()->access($this->currentUser, FALSE) ? $link->toRenderable() : ['#markup' => $definition->getLabel()],
          ];
        }

      }
    }

    return $build;
  }

  /**
   * Builds the response.
   */
  public function entityList(string $entity_type_id, Request $request): array {
    $storage = $this->entityTypeManager->getStorage($entity_type_id);
    $query = $request->query->all();
    $page = intval($query['page'] ?? 1);
    $ids = $storage->getQuery()->accessCheck(FALSE)->range(max(0, static::LIST_MAX * ($page - 1)), static::LIST_MAX + 1)->execute();
    $result_count = min(count($ids), static::LIST_MAX);

    sort($ids, SORT_NUMERIC);

    $build = [];
    $pager = [];

    if ($page > 1) {
      $pager['previous']['#markup'] = '<a href="?page=' . ($page - 1) . '">' . $this->t('&larr; Prev') . '</a>';
    }

    if (count($ids) > static::LIST_MAX) {
      if (isset($pager['previous'])) {
        $pager['separator']['#markup'] = ' | ';
      }

      $pager['next']['#markup'] = '<a href="?page=' . ($page + 1) . '">' . $this->t('Next &rarr;') . '</a>';
    }

    $build['pager_top'] = [
      '#type' => 'inline_template',
      '#template' => '<div style="margin-bottom:1em">Showing {{ count }} results for Page {{ page }}.</div><div style="margin-bottom:1em">{{ pager }}</div>',
      '#context' => [
        'pager' => $pager,
        'page' => $page,
        'count' => $result_count,
      ],
    ];

    $build['content'] = [
      '#theme' => 'plain_table',
      '#data' => [],
    ];

    $build['pager_bottom'] = [
      '#type' => 'inline_template',
      '#template' => '<div style="margin-top:1em">{{ pager }}</div>',
      '#context' => [
        'pager' => $pager,
      ],
    ];

    foreach ($storage->loadMultiple(array_slice($ids, 0, static::LIST_MAX)) as $entity) {
      $label = $entity->label() ?: $this->t('@type @id', [
        '@type' => $entity->getEntityType()->getLabel(),
        '@id' => $entity->id(),
      ]);

      $link = Link::createFromRoute($label, 'monarch_advanced_entity_usage.usage_list', ['entity_type_id' => $entity->getEntityTypeId(), 'entity_id' => $entity->id()], ['attributes' => ['target' => '_blank']]);
      $build['content']['#data'][$entity->id()] = $this->hasBundles($entity->getEntityTypeId()) ? [
        '#items' => [
          $entity->bundle(),
          $link->getUrl()->access($this->currentUser, FALSE) ? $link->toRenderable() : ['#markup' => $label],
        ],
      ] : ($link->getUrl()->access($this->currentUser, FALSE) ? $link->toRenderable() : ['#markup' => $label]);
    }

    return $build;
  }

  /**
   * Builds the response.
   */
  public function usageList(string $entity_type_id, string $entity_id, Request $request): array {
    $query = $request->query->all();
    $find_text_references = in_array($query['text'] ?? NULL, ['1', 't', 'true', 'y', 'yes']);
    $entity = $this->entityTypeManager->getStorage($entity_type_id)->load($entity_id);

    $build['content']['entity_info']['table'] = [
      '#theme' => 'plain_table',
      '#data' => [],
    ];

    $build['content']['entity_info']['table']['#data']['Type'] = $entity->getEntityTypeId();

    if ($this->hasBundles($entity->getEntityTypeId())) {
      $build['content']['entity_info']['table']['#data']['Bundle'] = $entity->bundle();
    }

    $build['content']['entity_info']['table']['#data']['ID'] = $entity->id();

    if ($entity instanceof RevisionableInterface && $entity->getEntityType()->isRevisionable()) {
      $build['content']['entity_info']['table']['#data']['Revision'] = $entity->getRevisionId();
    }

    $links = [];

    foreach ($entity->getEntityType()->getLinkTemplates() as $link_type => $path) {
      if ($entity->hasLinkTemplate($link_type)) {
        $link = $entity->toLink($link_type, $link_type, [
            'attributes' => ['target' => '_blank'],
        ]);

        if ($link->getUrl()->access($this->currentUser, FALSE)) {
          $links[$link_type] = $link->toString();
        }
      }
    }

    if ($links) {
      ksort($links);

      $links = array_filter([
        'canonical' => $links['canonical'] ?? NULL,
        'edit-form' => $links['edit-form'] ?? NULL,
      ]) + $links;

      $build['content']['entity_info']['links']['#type'] = 'inline_template';
      $build['content']['entity_info']['links']['#template'] = '';
      $build['content']['entity_info']['links'] = [
        '#type' => 'inline_template',
        '#template' => '<p>Links: {{ links }}</p>',
        '#context' => [
          'links' => Markup::create(implode(' | ', $links)),
        ],
      ];
    }

    if (!$find_text_references) {
      $build['content']['text_references_link']['#markup'] = $this->t('<p><a href="?text=true">Click here to also find references in text fields. (slow)</a></p>');
    }

    $references = $this->advancedEntityUsage->getReferences($entity, $find_text_references);

    foreach ($references as $parent_entity_type_id => $parent_entity_ids) {
      $parent_entity_definition = $this->entityTypeManager->getDefinition($parent_entity_type_id);
      $storage = $this->entityTypeManager->getStorage($parent_entity_type_id);

      ksort($parent_entity_ids, SORT_NUMERIC);

      foreach ($parent_entity_ids as $parent_entity_id => $revision_ids) {
        $parent_entity = $storage->load($parent_entity_id);

        if (!$parent_entity) {
          continue;
        }

        $build['content']['reference_link'][$parent_entity_type_id]['#prefix'] = $this->t('<h2>@label referencing this entity</h2>', [
          '@label' => ucwords($parent_entity_definition->getPluralLabel()),
        ]);

        $label = NULL;
        $includes_current_revision = FALSE;
        $revision_count = 1;

        if ($parent_entity instanceof RevisionableInterface) {
          $revision_count = count($revision_ids);

          if ($revision_ids[$parent_entity->getRevisionId()] ?? NULL) {
            $includes_current_revision = TRUE;
            $revision_count--;
          }
        }

        $label = $parent_entity->label() ?: $this->t('@type @id', [
          '@type' => $parent_entity_definition->getLabel(),
          '@id' => $parent_entity->id(),
        ]);

        if ($includes_current_revision) {
          if ($revision_count > 0) {
            $label = $revision_count > 1 ? $this->t('@label (current and @count revisions)', [
              '@label' => $label,
              '@count' => $revision_count,
            ]) : $this->t('@label (current and 1 revision)', [
              '@label' => $label,
            ]);
          }
        }
        else {
          $label = $revision_count > 1 ? $this->t('@label (@count revisions)', [
            '@label' => $label,
            '@count' => $revision_count,
          ]) : $this->t('@label (1 revision)', [
            '@label' => $label,
          ]);
        }

        $link = Link::createFromRoute($label, 'monarch_advanced_entity_usage.usage_list', ['entity_type_id' => $parent_entity->getEntityTypeId(), 'entity_id' => $parent_entity->id()], ['attributes' => ['target' => '_blank']]);

        $build['content']['reference_link'][$parent_entity->getEntityTypeId()][$parent_entity->id()] = [
          '#prefix' => '<div>',
          '#suffix' => '</div>',
          'parent_reference' => $link->getUrl()->access($this->currentUser, FALSE) ? $link->toRenderable() : ['#markup' => $label],
        ];
      }
    }

    $build['content']['reference_link'] ??= [
      '#prefix' => '<p>',
      '#suffix' => '</p>',
      'message' => [
        '#markup' => $this->t('No references found.'),
      ],
    ];

    return $build;
  }

}
