<?php

namespace Drupal\monarch_advanced_entity_usage;

use Drupal\Component\Utility\Html;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Sql\SqlEntityStorageInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldTypePluginManagerInterface;
use Drupal\Core\Field\Plugin\Field\FieldType\EntityReferenceItemBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\link\LinkItemInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\text\Plugin\Field\FieldType\TextItemBase;

/**
 * @todo Add class description.
 */
final class AdvancedEntityUsage {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The plugin.manager.field.field_type service.
   */
  protected FieldTypePluginManagerInterface $fieldTypeManager;

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * The path alias manager.
   */
  protected ?AliasManagerInterface $aliasManager;

  /**
   * The module_handler service.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * Constructs an AdvancedEntityUsage object.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, EntityFieldManagerInterface $entityFieldManager, FieldTypePluginManagerInterface $fieldTypeManager, Connection $database, ModuleHandlerInterface $moduleHandler, ?AliasManagerInterface $aliasManager = NULL) {
    $this->entityTypeManager = $entityTypeManager;
    $this->entityFieldManager = $entityFieldManager;
    $this->fieldTypeManager = $fieldTypeManager;
    $this->database = $database;
    $this->aliasManager = $aliasManager;
    $this->moduleHandler = $moduleHandler;
  }

  public function getFieldMap() {
    $map = $this->entityFieldManager->getFieldMap();

    $this->moduleHandler->alter('advanced_entity_usage_map', $map);

    return $map;
  }

  /**
   * @todo Add method description.
   */
  public function getReferences(EntityInterface $entity, bool $find_text_references = FALSE) : array {
    $references = [];

    $map = $this->getFieldMap();

    $uris = [];

    if ($entity->hasLinkTemplate('canonical')) {
      $url = $entity->toUrl('canonical');
      $uris[] = $url->toUriString();
      $default_path = $url->setAbsolute(FALSE)->toString();
      $default_path = $this->aliasManager ? $this->aliasManager->getPathByAlias($default_path) : $default_path;
      $alias = $this->aliasManager ? $this->aliasManager->getAliasByPath($default_path) : NULL;

      foreach ([
        $default_path,
        $alias,
      ] as $path) {
        if ($path) {
          $uris[] = 'internal:' . $path;
          $uris[] = 'base:' . ltrim($path, '/');
        }
      }

      foreach (['', '/'] as $suffix) {
        $uris[] = rtrim($url->setAbsolute(TRUE)->toString(), '/') . $suffix;
      }

      foreach ($uris as $uri) {
        $uris[] = $uri . '?%';
        $uris[] = $uri . '#%';
      }

      $uris[] = 'entity:' . $entity->getEntityTypeId() . '/' . $entity->id();
      $this->moduleHandler->alter('advanced_entity_usage_entity_uris', $uris, $entity);
      $uris = array_unique($uris);
    }

    foreach (array_keys($map) as $parent_entity_type_id) {
      $parent_entity_definition = $this->entityTypeManager->getDefinition($parent_entity_type_id);
      $storage = $this->entityTypeManager->getStorage($parent_entity_type_id);

      if ($storage instanceof SqlEntityStorageInterface) {
        if (!empty($definitions = $this->getFieldStorageDefinitionsByInterface($parent_entity_type_id, EntityReferenceItemBase::class))) {
          foreach ($definitions as $field_name => $definition) {
            if ($definition->getSetting('target_type') === $entity->getEntityTypeId()) {
              $query = $storage->getQuery()->accessCheck(FALSE)->condition($field_name . '.target_id', $entity->id());
  
              if ($parent_entity_definition->isRevisionable()) {
                $query->allRevisions();
              }
  
              $results = $query->execute();
  
              foreach ($results as $result_revision_id => $result_entity_id) {
                $references[$parent_entity_type_id][$result_entity_id][$result_revision_id] = $result_revision_id;
              }
            }
          }
        }

        if ($uris && !empty($definitions = $this->getFieldStorageDefinitionsByInterface($parent_entity_type_id, LinkItemInterface::class))) {
          foreach ($definitions as $field_name => $definition) {
            $query = $storage->getQuery()->accessCheck(FALSE);

            $orcondition = $query->orConditionGroup();

            foreach ($uris as $uri) {
              $orcondition->condition($field_name . '.uri', $uri, 'LIKE');
            }

            $query->condition($orcondition);

            if ($parent_entity_definition->isRevisionable()) {
              $query->allRevisions();
            }

            $results = $query->execute();

            foreach ($results as $result_revision_id => $result_entity_id) {
              $references[$parent_entity_type_id][$result_entity_id][$result_revision_id] = $result_revision_id;
            }
          }
        }

        if ($find_text_references && !empty($definitions = $this->getFieldStorageDefinitionsByInterface($parent_entity_type_id, TextItemBase::class))) {
          foreach ($definitions as $field_name => $definition) {
            $columns = $definition->getColumns();

            foreach (['value', 'summary'] as $property) {
              if (!isset($columns[$property])) {
                continue;
              }

              $possible_entity_query = $storage->getQuery()->accessCheck(FALSE)
                ->condition($field_name . '.' . $property, '%' . $this->database->escapeLike('data-entity-type="' . $entity->getEntityTypeId() . '"') . '%', 'LIKE')
                ->condition($field_name . '.' . $property, '%' . $this->database->escapeLike('data-entity-uuid="' . $entity->uuid() . '"') . '%', 'LIKE');

              if ($parent_entity_definition->isRevisionable()) {
                $possible_entity_query->allRevisions();
              }

              $possible_entity_ids = $possible_entity_query->execute();

              if ($possible_entity_ids) {
                foreach ($possible_entity_ids as $possible_entity_revision_id => $possible_entity_id) {
                  if ($parent_entity_definition->isRevisionable()) {
                    /** @var \Drupal\Core\Entity\RevisionableStorageInterface $storage */
                    $possible_entity = $storage->loadRevision($possible_entity_revision_id);
                  }
                  else {
                    $possible_entity = $storage->load($possible_entity_id);
                  }

                  if (!$possible_entity) {
                    continue;
                  }

                  foreach ($possible_entity->{$field_name} as $field_item) {
                    $text = $field_item->{$property};
                    $dom = Html::load($text);
                    $xpath = new \DOMXPath($dom);
                    foreach ($xpath->query('//*[@data-entity-type="' . $entity->getEntityTypeId() . '" and @data-entity-uuid="' . $entity->uuid() . '"]') as $node) {
                      $references[$parent_entity_type_id][$possible_entity_id][$possible_entity_revision_id] = $result_revision_id;
                    }
                  }
                }
              }
            }
          }
        }
      }
    }

    return $references;
  }

    /**
   * Determines the formatted text fields on an entity.
   *
   * A field type is considered to provide formatted text if its class is a
   * subclass of Drupal\text\Plugin\Field\FieldType\TextItemBase.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   An entity whose fields to analyze.
   *
   * @return \Drupal\Core\Field\FieldStorageDefinitionInterface[]
   *   The names of the fields on this entity that support formatted text.
   */
  protected function getFieldStorageDefinitionsByInterface(string $parent_entity_type_id, string ...$class_name) {
    $field_definitions = $this->entityFieldManager->getFieldStorageDefinitions($parent_entity_type_id);

    $ret = [];

    foreach ($field_definitions as $field_name => $definition) {
      if (isset($ret[$field_name])) {
        continue;
      }

      $type = $definition->getType();
      $plugin_class = $this->fieldTypeManager->getPluginClass($type);

      foreach ($class_name as $class_name_entry) {
        if (is_subclass_of($plugin_class, $class_name_entry)) {
          $ret[$field_name] = $definition;
          break;
        }
      }
    }

    return $ret;
  }

}
