<?php

namespace Drupal\monarch_ajax_content\Element;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\Element\RenderElement;
use Drupal\Core\Routing\RouteProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a render element to display an ajax content.
 *
 * Properties:
 * - #foo: Property description here.
 *
 * Usage Example:
 * @code
 * $build['ajax_content'] = [
 *   '#type' => 'ajax_content',
 *   '#foo' => 'Some value.',
 * ];
 * @endcode
 *
 * @RenderElement("ajax_content")
 */
final class AjaxContentElement extends RenderElement implements ContainerFactoryPluginInterface {

  /**
   * The route provider service.
   */
  private readonly RouteProviderInterface $routeProvider;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);

    $instance->routeProvider = $container->get('router.route_provider');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getInfo(): array {
    return [
      // @DCG
      // If the element is supposed to return a large piece of markup consider
      // defining a theme function for it.
      '#pre_render' => [
        [self::class, 'preRenderEntityElement'],
      ],
      '#id' => NULL,
      '#tag' => 'div',
      '#lazy' => TRUE,
      '#attributes' => [],
    ];
  }

  /**
   * Ajax content element pre render callback.
   *
   * @param array $element
   *   The element info.
   *
   * @return array
   *   The modified element.
   */
  public static function preRenderEntityElement(array $element): array {
    if (
      $element['#id'] &&
      ($plugin = \Drupal::service('plugin.manager.ajax_content')->createInstance($element['#id']))
    ) {
      $plugin->buildWrapper($element);

      $lazy = $element['#lazy'] ?? TRUE;
      $plugin_lazy = $plugin->pluginDefinition['lazy'] ?? TRUE;

      if ($lazy && $plugin_lazy) {
        return $element;
      }

      $element['wrapper']['content'] = $plugin->build(NULL, NULL);

      if (method_exists($plugin, 'preRender')) {
        $plugin->preRender($element['wrapper']['content'], TRUE);
      }

      $element['wrapper']['#attributes']['data-ajax-content-element-init'] = 'static';
      $element['wrapper']['#attributes']['data-ajax-content-element-static-init'] = 'true';
    }

    return $element;
  }

}
