<?php

namespace Drupal\monarch_algolia_index;

use Algolia\AlgoliaSearch\SearchClient;
use Algolia\AlgoliaSearch\SearchIndex;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\RendererInterface;

/**
 * The algolia_index.indexer service.
 */
final class Indexer {

  const CONFIG_NAME = 'monarch_algolia_index.settings';

  /**
   * Constructs an Indexer object.
   */
  public function __construct(
    private readonly Connection $database,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly RendererInterface $renderer,
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly LockBackendInterface $lock,
    private readonly MessengerInterface $messenger,
  ) {}

  /**
   * The module config.
   */
  public function config() : ImmutableConfig {
    return $this->configFactory->get(static::CONFIG_NAME);
  }

  /**
   * The appId.
   */
  public function getAppId() : string {
    return $this->config()->get('appid') ?: '';
  }

  /**
   * The apiKey.
   */
  public function getApiKey() : string {
    return $this->config()->get('apikey') ?: '';
  }

  /**
   * The SearchClient.
   */
  public function client() : ?SearchClient {
    if (($appid = $this->getAppId()) && ($apikey = $this->getApiKey())) {
      return SearchClient::create($appid, $apikey);
    }

    return NULL;
  }

  /**
   * The index name.
   */
  public function getIndexName() : string {
    return $this->config()->get('index') ?: '';
  }

  /**
   * The site name.
   */
  public function getSiteName() : string {
    return str_replace('|', '_', $this->config()->get('site') ?: 'default');
  }

  /**
   * The environment name.
   */
  public function getEnvironmentName() : string {
    return str_replace('|', '_', $this->config()->get('environment') ?: 'dev');
  }

  /**
   * The index.
   */
  public function getIndex() : ?SearchIndex {
    if (($index_name = $this->getIndexName()) && ($client = $this->client())) {
      return $client->initIndex($index_name);
    }

    return NULL;
  }

  /**
   * Are we allowed to index?
   */
  public function canIndex() : bool {
    return (bool) ($this->config()->get('allow') ?? FALSE);
  }

  /**
   * Index an item.
   */
  public function indexItem(string $entity_type_id, string $bundle_id, string $entity_id, ?string $status = 'add', bool $break_preindex = TRUE) {
    if ($break_preindex) {
      $this->lock->release('algolia_index_preindexing');
    }

    $this->database->merge('algolia_index_queue')
      ->key([
        'type' => $entity_type_id,
        'bundle' => $bundle_id,
        'id' => $entity_id,
      ])
      ->fields([
        'type' => $entity_type_id,
        'bundle' => $bundle_id,
        'id' => $entity_id,
        'status' => $status,
        'timestamp' => time(),
      ])
      ->execute();
  }

  /**
   * Index all content.
   */
  public function indexAllNodes(array $bundles = NULL, bool $break_preindex = TRUE) {
    if ($break_preindex) {
      $this->lock->release('algolia_index_preindexing');
    }

    $query = $this->database->select('node', 'n')
      ->fields('n', ['nid', 'type']);

    if (!empty($bundles)) {
      $query->condition('type', $bundles, 'IN');
    }

    foreach ($query->execute() as $node) {
      $this->indexItem('node', $node->type, $node->nid, 'add', FALSE);
    }
  }

  /**
   * Index all content.
   */
  public function unindexAllNodes(array $bundles = NULL, bool $break_preindex = TRUE) {
    if ($break_preindex) {
      $this->lock->release('algolia_index_preindexing');
    }

    $query = $this->database->select('algolia_index_queue', 'n')
      ->fields('n', ['id', 'bundle'])
      ->orderBy('timestamp')
      ->condition('indexed', 0, '!=')
      ->condition('type', 'node');

    if (!empty($bundles)) {
      $query->condition('bundle', $bundles, 'IN');
    }

    foreach ($query->execute() as $node) {
      $this->indexItem('node', $node->bundle, $node->id, 'remove', FALSE);
    }
  }

  /**
   * Index an item.
   */
  public function updateIndexItem(string $entity_type_id, string $bundle_id, string $entity_id, array $fields, bool $break_preindex = TRUE) {
    if ($break_preindex) {
      $this->lock->release('algolia_index_preindexing');
    }

    $fields['timestamp'] = $fields['timestamp'] ?? time();

    $this->database->merge('algolia_index_queue')
      ->key([
        'type' => $entity_type_id,
        'bundle' => $bundle_id,
        'id' => $entity_id,
      ])
      ->fields($fields)
      ->execute();
  }

  /**
   * Entity update event.
   */
  public function onEntityIndex(EntityInterface $entity, string $event) {
    $entity_type_id = $entity->getEntityTypeId();
    $entity_bundle_id = $entity->bundle();
    $entity_id = $entity->id();
    $types_to_index = $this->config()->get('types');
    $type_bundle = $entity->getEntityTypeId() . '|' . $entity->bundle();

    if ($types_to_index[$type_bundle] ?? NULL) {
      $this->indexItem($entity_type_id, $entity_bundle_id, $entity_id, $event === 'delete' ? 'remove' : 'add');
    }
  }

  /**
   * Get Algolia Object ID.
   */
  public function getAlgoliaObjectId(string $entity_type, string $id, string $language = 'en') {
    return $this->getSiteName() . '|' . $this->getEnvironmentName() . "|entity:$entity_type/$id:$language";
  }

  /**
   * Truncate text while retaining words.
   */
  public function textTruncate(string $text, $max = 40000) : string {
    if (strlen($text) <= $max) {
      return $text;
    }

    $text = substr($text, 0, $max + 1);
    $pos = strrpos($text, ' ');
    return substr($text, 0, $pos);
  }

  /**
   * Process the indexing queue.
   */
  public function processQueue(int $limit = 10) : int {
    try {
      if ($limit < 1) {
        $this->messenger->addWarning('Limit must be > 0.');
        return 0;
      }

      if (!$this->canIndex()) {
        $this->messenger->addWarning('Not allowed to index!');
        return 0;
      }

      if (!$this->lock->acquire('algolia_index_indexing', 900)) {
        $this->messenger->addWarning('Unable to acquire lock: algolia_index_indexing');
        return 0;
      }

      $index = $this->getIndex();

      if ($this->lock->acquire('algolia_index_preindexing', 900)) {
        $this->moduleHandler->invokeAll('algolia_preindex');
      }

      $site = $this->getSiteName();

      $remove_entities = $this->database->select('algolia_index_queue', 'iq')
        ->orderBy('timestamp')
        ->condition('status', 'remove')
        ->condition('indexed', '0', '!=')
        ->fields('iq', ['type', 'bundle', 'id'])
        ->execute()
        ->fetchAll();

      $remove_entity_ids = [];

      foreach ($remove_entities as $i => $n) {
        $remove_entity_ids[] = $this->getAlgoliaObjectId($n->type, $n->id);
      }

      if (!empty($remove_entity_ids)) {
        if ($index->deleteObjects($remove_entity_ids)) {
          foreach ($remove_entities as $i => $n) {
            $this->updateIndexItem($n->type, $n->bundle, $n->id, [
              'status' => NULL,
              'indexed' => 0,
            ], FALSE);
          }
        }
      }

      $add_entity_ids = $this->database->select('algolia_index_queue', 'iq')
        ->orderBy('timestamp')
        ->condition('status', 'add')
        ->fields('iq', ['type', 'bundle', 'id'])
        ->range(0, $limit)
        ->execute()
        ->fetchAll();

      if (empty($add_entity_ids)) {
        return 0;
      }

      $add_entities = [];

      foreach ($add_entity_ids as $i => $n) {
        $add_entity = \Drupal::entityTypeManager()->getStorage($n->type)->load($n->id);
        $fields = [];

        $fields['should_index'] = TRUE;
        $fields['title'] = trim($add_entity->label());
        $fields['path'] = $add_entity->toUrl()->toString();

        $this->moduleHandler->alter('algolia_index_fields', $fields, $add_entity);

        if ($fields['should_index'] ?? NULL) {
          unset($fields['should_index']);

          if (!isset($fields['rendered_text'])) {
            $vb = \Drupal::entityTypeManager()->getViewBuilder($n->type);
            $render = $vb->view($add_entity);
            $rendered_text = strip_tags((string) $this->renderer->renderPlain($render));
            $rendered_text = trim(preg_replace("/\s+/", " ", $rendered_text));
            $fields['rendered_text'] = $this->textTruncate(html_entity_decode($rendered_text));
          }

          $fields = [
            'objectID' => $this->getAlgoliaObjectId($add_entity->getEntityTypeId(), $add_entity->id()),
            'site' => $site,
            'environment' => $this->getEnvironmentName(),
            'type' => $add_entity->getEntityTypeId(),
            'bundle' => $add_entity->bundle(),
          ] + $fields;

          $add_entities[] = $fields;
          $fields = [];
        }
        else {
          $this->indexItem($n->type, $n->bundle, $n->id, 'remove', FALSE);
        }
      }

      if (!empty($add_entities)) {
        if ($index->saveObjects($add_entities)) {
          $add_entities = [];
          foreach ($add_entity_ids as $i => $n) {
            $this->updateIndexItem($n->type, $n->bundle, $n->id, [
              'status' => NULL,
              'indexed' => 1,
            ], FALSE);
          }
        }
      }

      return count($add_entity_ids);
    }
    catch (\Throwable $err) {
      throw $err;
    }
    finally {
      $this->lock->release('algolia_index_indexing');
    }
  }

}
