<?php

namespace Drupal\monarch_algolia_index\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\monarch_algolia_index\Indexer;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Monarch Algolia Index form.
 */
final class IndexForm extends FormBase {

  /**
   * The database service.
   */
  private Connection $database;

  /**
   * The algolia_index.indexer service.
   */
  private Indexer $indexer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->database = $container->get('database');
    $instance->indexer = $container->get('algolia_index.indexer');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'monarch_algolia_index_batch';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    if ($this->indexer->canIndex()) {
      $form['actions'] = [
        '#type' => 'actions',
        'submit' => [
          '#type' => 'submit',
          '#value' => $this->t('Process Index Queue'),
        ],
      ];
    }
    else {
      $form['message']['#markup'] = $this->t('Indexing not enabled.');
    }

    return $form;
  }

  /**
   * Get the queue count.
   */
  public function getCount() : int {
    $this->database = $this->database ?? \Drupal::database();

    return $this->database->select('algolia_index_queue', 'i')
      ->condition('status', 'add')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $count = $this->getCount();

    if ($count > 0) {
      batch_set([
        'title' => $this->t('Indexing'),
        'operations' => [
          [
            [$this, 'batch'],
            [$count, time()],
          ],
        ],
      ]);
    }
  }

  /**
   * The batch processor.
   */
  public function batch($max, $start, &$context) {
    try {
      $this->indexer = $this->indexer ?? \Drupal::service('algolia_index.indexer');
      $processed = $this->indexer->processQueue();
      $count = min($max, max(0, $max - $this->getCount()));
      $context['finished'] = $count / $max;
      $context['message'] = $this->t('@count / @max (@time seconds, @processed processed)', [
        '@count' => $count,
        '@max' => $max,
        '@time' => time() - $start,
        '@processed' => $processed,
      ]);
    }
    catch (\Throwable $err) {
      $context['finished'] = 0;
      $context['message'] = $err->getMessage();
    }
  }

}
