<?php declare(strict_types = 1);

namespace Drupal\monarch_analytics\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Monarch Analytics Framework settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * The entity_type.manager service.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->entityTypeManager = $container->get('entity_type.manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'monarch_analytics_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['monarch_analytics.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();

    foreach ($roles as $id => $role) {
      $roles[$id] = $role->label();
    }

    $form['route_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Route Types to Track'),
      '#options' => [
        'admin' => $this->t('Admin'),
        'nonadmin' => $this->t('Non-Admin'),
      ],
      '#default_value' => $this->config('monarch_analytics.settings')->get('route_types') ?: [],
    ];

    $form['roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Roles to Track'),
      '#options' => $roles,
      '#default_value' => $this->config('monarch_analytics.settings')->get('roles') ?: [],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('monarch_analytics.settings')
      ->set('roles', $form_state->getValue('roles'))
      ->set('route_types', $form_state->getValue('route_types'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
