<?php

namespace Drupal\monarch_analytics\Plugin\RestApi;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Path\PathValidatorInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\monarch_analytics\MetricsEvent;
use Drupal\monarch_analytics\UrlUtility;
use Drupal\monarch_rest_plugin\RestApiPluginBase;
use Drupal\monarch_rest_plugin\RestApiRoute;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Monarch Metrics REST API.
 *
 * @RestApi(
 *   id = "monarch_analytics",
 * )
 */
final class UrlMetrics extends RestApiPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The path.validator service.
   */
  protected PathValidatorInterface $pathValidator;

  /**
   * The module_handler service.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The url_utility service.
   */
  protected UrlUtility $urlUtility;

  /**
   * The event_dispatcher service.
   */
  protected EventDispatcherInterface $eventDispatcher;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);

    $instance->pathValidator = $container->get('path.validator');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->urlUtility = $container->get('url_utility');
    $instance->eventDispatcher = $container->get('event_dispatcher');

    return $instance;
  }

  /**
   * Get a list of routes provided by this plugin.
   */
  public function getRoutes() : iterable {
    $route = ($this->pluginDefinition['route'] ?? explode('\\', get_class($this), 5)[4] ?? $this->pluginDefinition['id']);

    if (is_string($route)) {
      return [
        new RestApiRoute($route, [$this, '__invoke']),
        new RestApiRoute($route . '/View', [$this, 'view']),
        new RestApiRoute($route . '/Click', [$this, 'click']),
        new RestApiRoute($route . '/Show', [$this, 'show']),
        new RestApiRoute($route . '/Hide', [$this, 'hide']),
      ];
    }

    return [];
  }

  /**
   * The default rest API endpoint.
   *
   * The parameters are optional and can be in any order.
   */
  public function __invoke() {
    $ret = [];

    foreach (self::getRoutes() as $index => $route) {
      /** @var \Drupal\monarch_rest_plugin\RestApiRoute $route */
      if ($index > 0) {
        $ret[] = $route->getRoute();
      }
    }

    return $ret;
  }

  /**
   * The view rest API endpoint wrapper.
   *
   * The parameters are optional and can be in any order.
   */
  public function view(array $query = NULL, Request $request = NULL, AccountInterface $account = NULL) {
    $href = $query['href'] ?? NULL;

    if (is_null($href)) {
      return [
        'status' => FALSE,
        'message' => 'Must provide a \'href\' parameter.',
      ];
    }

    $headers = NULL;

    if ($code = ($query['code'] ?? NULL)) {
      $data = \Drupal::database()->select('monarch_analytics__server_vars', 't')->condition('hash', $code)->fields('t', ['data'])->execute()->fetchField();

      if (!empty($data)) {
        $headers = json_decode($data, TRUE);
      }
    }

    [$url, $href] = $this->urlUtility->getUrl($href, $request);

    if ($url) {
      $event = new MetricsEvent(
        'view',
        $href,
        $url,
        $url && $url->isRouted() ? $this->urlUtility->isAdminRoute($url->getRouteName()) : FALSE,
        $query['metadata'] ?? NULL,
        $account,
        $request,
        $headers,
      );

      $this->eventDispatcher->dispatch($event, MetricsEvent::VIEW);
      $this->moduleHandler->invokeAll('url_metrics_view_event', [$event]);

      if ($referer = ($headers['referer'][0] ?? NULL)) {
        $url_info = parse_url($referer);

        if ($request->getHost() !== $url_info['host']) {
          $event = new MetricsEvent(
            'landing',
            $href,
            $url,
            $url && $url->isRouted() ? $this->urlUtility->isAdminRoute($url->getRouteName()) : FALSE,
            $query['metadata'] ?? NULL,
            $account,
            $request,
            $headers,
          );

          $this->eventDispatcher->dispatch($event, MetricsEvent::LANDING);
          $this->moduleHandler->invokeAll('url_metrics_landing_event', [$event]);
        }
      }
    }

    return [
      'status' => TRUE,
      'message' => 'OK',
    ];
  }

  /**
   * The click rest API endpoint wrapper.
   *
   * The parameters are optional and can be in any order.
   */
  public function click(array $query = NULL, Request $request = NULL, AccountInterface $account = NULL) {
    $href = $query['href'] ?? NULL;

    if (is_null($href)) {
      return [
        'status' => FALSE,
        'message' => 'Must provide a \'href\' parameter.',
      ];
    }

    $headers = NULL;

    if ($code = ($query['code'] ?? NULL)) {
      $data = \Drupal::database()->select('monarch_analytics__server_vars', 't')->condition('hash', $code)->fields('t', ['data'])->execute()->fetchField();

      if (!empty($data)) {
        $headers = json_decode($data, TRUE);
      }
    }

    [$url, $href] = $this->urlUtility->getUrl($href, $request);

    if ($url) {
      $parent_href = $query['metadata']['parent']['href'] ?? NULL;

      if ($parent_href) {
        [$parent_href] = $this->urlUtility->urlBreakdown($parent_href, $request);
        $query['metadata']['parent']['href'] = $parent_href;
      }

      $event = new MetricsEvent(
        'click',
        $href,
        $url,
        $url && $url->isRouted() ? $this->urlUtility->isAdminRoute($url->getRouteName()) : FALSE,
        $query['metadata'] ?? NULL,
        $account,
        $request,
        $headers,
      );

      $this->eventDispatcher->dispatch($event, MetricsEvent::CLICK);
      $this->moduleHandler->invokeAll('url_metrics_click_event', [$event]);
    }

    return [
      'status' => TRUE,
      'message' => 'OK',
    ];
  }

  /**
   * The show rest API endpoint wrapper.
   *
   * The parameters are optional and can be in any order.
   */
  public function show(array $query = NULL, Request $request = NULL, AccountInterface $account = NULL) {
    $href = $query['href'] ?? NULL;

    if (is_null($href)) {
      return [
        'status' => FALSE,
        'message' => 'Must provide a \'href\' parameter.',
      ];
    }

    $headers = NULL;

    if ($code = ($query['code'] ?? NULL)) {
      $data = \Drupal::database()->select('monarch_analytics__server_vars', 't')->condition('hash', $code)->fields('t', ['data'])->execute()->fetchField();

      if (!empty($data)) {
        $headers = json_decode($data, TRUE);
      }
    }

    [$url, $href] = $this->urlUtility->getUrl($href, $request);

    if ($url) {
      $event = new MetricsEvent(
        'show',
        $href,
        $url,
        $url && $url->isRouted() ? $this->urlUtility->isAdminRoute($url->getRouteName()) : FALSE,
        $query['metadata'] ?? NULL,
        $account,
        $request,
        $headers,
      );

      $this->eventDispatcher->dispatch($event, MetricsEvent::SHOW);
      $this->moduleHandler->invokeAll('url_metrics_show_event', [$event]);
    }

    return [
      'status' => TRUE,
      'message' => 'OK',
    ];
  }

  /**
   * The hide rest API endpoint wrapper.
   *
   * The parameters are optional and can be in any order.
   */
  public function hide(array $query = NULL, Request $request = NULL, AccountInterface $account = NULL) {
    $href = $query['href'] ?? NULL;

    if (is_null($href)) {
      return [
        'status' => FALSE,
        'message' => 'Must provide a \'href\' parameter.',
      ];
    }

    $headers = NULL;

    if ($code = ($query['code'] ?? NULL)) {
      $data = \Drupal::database()->select('monarch_analytics__server_vars', 't')->condition('hash', $code)->fields('t', ['data'])->execute()->fetchField();

      if (!empty($data)) {
        $headers = json_decode($data, TRUE);
      }
    }

    [$url, $href] = $this->urlUtility->getUrl($href, $request);

    if ($url) {
      $event = new MetricsEvent(
        'hide',
        $href,
        $url,
        $url && $url->isRouted() ? $this->urlUtility->isAdminRoute($url->getRouteName()) : FALSE,
        $query['metadata'] ?? NULL,
        $account,
        $request,
        $headers,
      );

      $this->eventDispatcher->dispatch($event, MetricsEvent::HIDE);
      $this->moduleHandler->invokeAll('url_metrics_hide_event', [$event]);
    }

    return [
      'status' => TRUE,
      'message' => 'OK',
    ];
  }
}
