<?php

namespace Drupal\monarch_analytics;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Path\PathValidatorInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * The metrics service.
 */
final class UrlUtility {

  /**
   * A list of all routes that typically should be admin routes.
   */
  private array $adminRoutes = [];

  /**
   * Constructs a Metrics object.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly RequestStack $requestStack,
    private readonly PathValidatorInterface $pathValidator,
    private readonly RouteProviderInterface $routeProvider,
  ) {
    \Drupal::pathValidator();

    foreach ($this->entityTypeManager->getDefinitions() as $key => $definition) {
      $this->adminRoutes["$key.add"] = "$key.add";
      $this->adminRoutes["entity.$key.add_page"] = "entity.$key.add_page";
      $this->adminRoutes["entity.$key.edit_form"] = "entity.$key.edit_form";
      $this->adminRoutes["entity.$key.delete_form"] = "entity.$key.delete_form";
      $this->adminRoutes["entity.$key.collection"] = "entity.$key.collection";
      $this->adminRoutes["entity.$key.version_history"] = "entity.$key.version_history";
      $this->adminRoutes["entity.$key.entityqueue"] = "entity.$key.entityqueue";
    }
  }

  /**
   * Modify the url to make it easier to get a valid url.
   */
  public function urlBreakdown(string $url, Request $request = NULL) : array {
    $request = $request ?? $this->requestStack->getCurrentRequest();
    $url_info = parse_url($url);

    $external = !(empty($url_info['host'] ?? NULL) || ($request && $url_info['host'] === $request->getHost()));
    return [$external ? $url : $url_info['path'], $external];
  }

  /**
   * Gets an aggressively matched url.
   */
  public function getUrl(string $url, Request $request = NULL) : array {
    [$url, $external] = $this->urlBreakdown($url, $request);

    try {
      return [
        $this->pathValidator->getUrlIfValid($url) ?: Url::fromUri($external ? $url : 'base:/' . ltrim($url, '/')),
        $url,
      ];
    }
    catch (\Throwable $err) {
      return [NULL, NULL];
    }
  }

  /**
   * Checks if a route is an admin route.
   */
  public function isAdminRoute(string $route_name) {
    try {
      if ($route_name && ($this->adminRoutes[$route_name] ?? NULL)) {
        return TRUE;
      }

      $route = $this->routeProvider->getRouteByName($route_name);

      if ($route && $route->getOption('_admin_route')) {
        return TRUE;
      }
    }
    catch (\Throwable $err) {
      // Ignore.
    }

    return FALSE;
  }

}
