<?php

namespace Drupal\monarch_bootstrap\Plugin\Preprocess;

use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Base class for template preprocess plugins.
 */
abstract class PreprocessBase extends PluginBase implements PreprocessInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The theme preprocess hook invoked.
   *
   * @var string
   */
  protected $hook;

  /**
   * The theme preprocess hook info array from the theme registry.
   *
   * @var mixed[]
   */
  protected $info;

  /**
   * The theme manager service.
   *
   * @var \Drupal\Core\Theme\ThemeManagerInterface
   */
  protected $themeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->themeManager = $container->get('theme.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function preprocess(array &$variables, $hook, array $info) {
    $this->hook = $hook;
    $this->info = $info;
  }

  /**
   * Get the active theme object.
   *
   * @return \Drupal\Core\Theme\ActiveTheme
   *   The active theme.
   */
  public function getActiveTheme() {
    return $this->themeManager->getActiveTheme();
  }

  /**
   * Get all theme settings for the current theme.
   *
   * @param string $setting_name
   *   The name of the setting to be retrieved.
   */
  public function getThemeSetting($setting_name) {
    $theme_name = $this->getActiveTheme()->getName();
    return theme_get_setting($setting_name, $theme_name);
  }

  /**
   * Replace default attribute classes with bootstrap versions.
   *
   * @param string[] $class_map
   *   An associative array of class replacements. Empty replacements
   *   will be removed.
   * @param array $variables
   *   The preprocess variables array.
   */
  public function replaceClasses(array $class_map, array &$variables) {
    if (!isset($variables['attributes']['class'])) {
      return;
    }

    foreach ($variables['attributes']['class'] as $delta => $class) {
      if (isset($class_map[$class])) {
        if (empty($class_map[$class])) {
          unset($variables['attributes']['class'][$delta]);
        }
        else {
          $variables['attributes']['class'][$delta] = $class_map[$class];
        }
      }
    }
  }

}
