<?php

namespace Drupal\monarch_bootstrap\Plugin;

use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * Base class for plugin managers.
 */
class PluginManager extends DefaultPluginManager {

  /**
   * The theme handler to check if theme exists.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * The theme manager to invoke alter hooks.
   *
   * @var \Drupal\Core\Theme\ThemeManager
   */
  protected $themeManager;

  /**
   * Creates the discovery object.
   *
   * @param string|bool $subdir
   *   The plugin's subdirectory, for example Plugin/views/filter.
   * @param string|null $plugin_interface
   *   (optional) The interface each plugin should implement.
   * @param string $plugin_definition_annotation_name
   *   (optional) Name of the annotation that contains the plugin definition.
   *   Defaults to 'Drupal\Component\Annotation\Plugin'.
   */
  public function __construct($subdir, $plugin_interface = NULL, $plugin_definition_annotation_name = 'Drupal\Component\Annotation\Plugin') {
    $this->themeHandler = \Drupal::service('theme_handler');
    $this->themeManager = \Drupal::service('theme.manager');

    $theme_list = $this->themeHandler->listInfo();
    $theme_name = $this->themeHandler->getDefault();
    $theme = $this->themeHandler->getTheme($theme_name);

    $ancestry = $this->themeHandler->getBaseThemes($theme_list, $theme_name);

    // Get namespaces and paths for all themes in the ancestry.
    $namespaces = [];
    foreach (array_keys($ancestry) as $ancestor_name) {
      $ancestor = $this->themeHandler->getTheme($ancestor_name);
      $namespaces['Drupal\\' . $ancestor->getName()] = [DRUPAL_ROOT . '/' . $ancestor->getPath() . '/src'];
    }
    $namespaces['Drupal\\' . $theme->getName()] = [DRUPAL_ROOT . '/' . $theme->getPath() . '/src'];
    $this->namespaces = new \ArrayObject($namespaces);

    $this->subdir = $subdir;
    $this->pluginDefinitionAnnotationName = $plugin_definition_annotation_name;
    $this->pluginInterface = $plugin_interface;
  }

  /**
   * {@inheritdoc}
   */
  protected function findDefinitions() {
    $definitions = parent::findDefinitions();
    $this->sortDefinitions($definitions);
    return $definitions;
  }

  /**
   * {@inheritdoc}
   */
  protected function providerExists($provider) {
    return $this->themeHandler->themeExists($provider);
  }

  /**
   * Sorts the plugin definitions.
   *
   * @param array $definitions
   *   The unsorted plugin definitions, passed by reference.
   */
  protected function sortDefinitions(array &$definitions) {
    uasort($definitions, [
      '\Drupal\Component\Utility\SortArray', 'sortByWeightElement',
    ]);
  }

}
