<?php

namespace Drupal\monarch_bootstrap\Plugin\Alter\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\monarch_bootstrap\Plugin\Alter\AlterBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form alter plugin for the system_theme_settings form.
 *
 * @MonarchAlter("system_theme_settings")
 */
class SystemThemeSettingsAlter extends AlterBase {

  /**
   * The entity_type.manager service.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->entityTypeManager = $container->get('entity_type.manager');

    return $instance;
  }

  /**
   * Return an associative array of the custom menus names.
   *
   * @param bool $all
   *   (optional) If FALSE return only user-added menus, or if TRUE also include
   *   the menus defined by the system. Defaults to TRUE.
   *
   * @return array
   *   An array with the machine-readable names as the keys, and human-readable
   *   titles as the values.
   */
  protected function getMenus($all = TRUE) {
    /** @var \Drupal\system\MenuInterface[]|null $custom_menus */
    if ($custom_menus = $this->entityTypeManager->getStorage('menu')->loadMultiple()) {
      if (!$all) {
        $custom_menus = array_diff_key($custom_menus, [
          'tools' => 'Tools',
          'admin' => 'Administration',
          'account' => 'User account menu',
          'main' => 'Main navigation',
          'footer' => 'Footer menu',
        ]);
      }
      foreach ($custom_menus as $menu_name => $menu) {
        $custom_menus[$menu_name] = $menu
          ->label();
      }
      asort($custom_menus);
    }
    return $custom_menus;
  }

  /**
   * {@inheritdoc}
   */
  public function alter(&$data, &$context1 = NULL, &$context2 = NULL) {
    $form = &$data;
    $form_state = $context1;
    $form_id = $context2;

    $theme_name = $this->getActiveTheme()->getName();

    $form['bootstrap'] = [
      '#type' => 'vertical_tabs',
      '#prefix' => '<h2><small>' . $this->t('Bootstrap Settings') . '</small></h2>',
      '#weight' => -10,
    ];
    $groups = [
      'regions' => $this->t('Regions'),
      'menus' => $this->t('Menus'),
      'components' => $this->t('Components'),
    ];
    foreach ($groups as $group => $title) {
      $form[$group] = [
        '#type' => 'details',
        '#title' => $title,
        '#group' => 'bootstrap',
        '#tree' => TRUE,
      ];
    }

    // Region settings.
    $regions = system_region_list($this->getActiveTheme()->getExtension());
    foreach ($regions as $region_key => $region) {
      $form['regions'][$region_key] = [
        '#type' => 'details',
        '#title' => $region,
      ];

      $form['regions'][$region_key]['container'] = [
        '#type' => 'select',
        '#title' => $this->t('@region container', ['@region' => $region]),
        '#options' => [
          'none' => $this->t('None'),
          'container' => $this->t('Default'),
          'container-fluid' => $this->t('Fluid'),
          'container-sm' => $this->t('Small'),
          'container-md' => $this->t('Medium'),
          'container-lg' => $this->t('Large'),
          'container-xl' => $this->t('X-Large'),
          'container-xxl' => $this->t('XX-Large'),
        ],
        '#default_value' => theme_get_setting('regions.' . $region_key . '.container', $theme_name),
        '#description' => $this->t('Select the type of container to use for this region. See <a href=":url">https://getbootstrap.com/docs/5.1/layout/containers</a>', [':url' => 'https://getbootstrap.com/docs/5.1/layout/containers']),
      ];

      $form['regions'][$region_key]['bg_color'] = [
        '#type' => 'select',
        '#title' => $this->t('@region background color', ['@region' => $region]),
        '#options' => $this->getColorClassOptions(),
        '#default_value' => theme_get_setting('regions.' . $region_key . '.bg_color', $theme_name),
        '#description' => $this->t('Select the background color for this region. Colors are definied in variables.scss. Use "None" to exclude bootstrap background classes and allow customization in scss. See <a href=":url">https://getbootstrap.com/docs/5.1/customize/color/</a>', [':url' => 'https://getbootstrap.com/docs/5.1/customize/color/']),
      ];

      $form['regions'][$region_key]['classes'] = [
        '#type' => 'textfield',
        '#title' => $this->t('@region additional classes', ['@region' => $region]),
        '#default_value' => theme_get_setting('regions.' . $region_key . '.classes', $theme_name),
        '#description' => $this->t('Add additional classes separated by a space for this region.'),
      ];
    }

    // Menu settings.
    $menus = $this->getMenus();
    foreach ($menus as $menu_key => $menu) {
      $form['menus'][$menu_key] = [
        '#type' => 'details',
        '#title' => $menu,
      ];

      $form['menus'][$menu_key]['position'] = [
        '#type' => 'select',
        '#title' => $this->t('@menu placement', ['@menu' => $menu]),
        '#options' => [
          'none' => $this->t('Default'),
          'fixed-top' => $this->t('Fixed Top'),
          'fixed-bottom' => $this->t('Fixed Bottom'),
          'sticky-top' => $this->t('Sticky Top'),
        ],
        '#default_value' => theme_get_setting('menus.' . $menu_key . '.position', $theme_name),
        '#description' => $this->t('Select the placement for this menu. See <a href=":url">https://getbootstrap.com/docs/5.1/components/navbar/#placement</a>', [':url' => 'https://getbootstrap.com/docs/5.1/components/navbar/#placement']),
      ];

      $form['menus'][$menu_key]['expand'] = [
        '#type' => 'select',
        '#title' => $this->t('@menu responsive behavior', ['@menu' => $menu]),
        '#options' => [
          'none' => $this->t('Default (vertical)'),
          'navbar-expand' => $this->t('Always Expand (horizontal)'),
          'navbar-expand-sm' => $this->t('Expand Small'),
          'navbar-expand-md' => $this->t('Expand Medium'),
          'navbar-expand-lg' => $this->t('Expand Large'),
          'navbar-expand-xl' => $this->t('Expand X-Large'),
          'navbar-expand-xxl' => $this->t('Expand XX-Large'),
        ],
        '#default_value' => theme_get_setting('menus.' . $menu_key . '.expand', $theme_name),
        '#description' => $this->t('Select the width when the menu should expand. See <a href=":url">https://getbootstrap.com/docs/5.1/components/navbar/#responsive-behaviors</a>', [':url' => 'https://getbootstrap.com/docs/5.1/components/navbar/#responsive-behaviors']),
      ];

      $form['menus'][$menu_key]['classes'] = [
        '#type' => 'textfield',
        '#title' => $this->t('@menu additional classes', ['@menu' => $menu]),
        '#default_value' => theme_get_setting('menus.' . $menu_key . '.classes', $theme_name),
        '#description' => $this->t('Add additional classes separated by a space for this menu.'),
      ];
    }

    // Component settings.
    $form['components']['card'] = [
      '#type' => 'details',
      '#title' => $this->t('Card'),
      'button_class' => [
        '#type' => 'select',
        '#title' => $this->t('Card button color'),
        '#options' => $this->getColorClassOptions('btn'),
        '#default_value' => theme_get_setting('components.card.button_class', $theme_name),
        '#description' => $this->t('Select the default button color for card elements. Colors are definied in variables.scss. Use "None" to exclude a bootstrap color class and allow customization in scss. See <a href=":url">https://getbootstrap.com/docs/5.1/customize/color/</a>', [':url' => 'https://getbootstrap.com/docs/5.1/customize/color/']),
      ],
    ];

    $form['components']['input'] = [
      '#type' => 'details',
      '#title' => $this->t('Input'),
      'submit_class' => [
        '#type' => 'select',
        '#title' => $this->t('Submit button color'),
        '#options' => $this->getColorClassOptions('btn'),
        '#default_value' => theme_get_setting('components.input.submit_class', $theme_name),
        '#description' => $this->t('Select the default button color for submit input elements. Colors are definied in variables.scss. Use "None" to exclude a bootstrap color class and allow customization in scss. See <a href=":url">https://getbootstrap.com/docs/5.1/customize/color/</a>', [':url' => 'https://getbootstrap.com/docs/5.1/customize/color/']),
      ],
      'button_class' => [
        '#type' => 'select',
        '#title' => $this->t('Default button color'),
        '#options' => $this->getColorClassOptions('btn'),
        '#default_value' => theme_get_setting('components.input.button_class', $theme_name),
        '#description' => $this->t('Select the default color for button elements. Colors are definied in variables.scss. Use "None" to exclude a bootstrap color class and allow customization in scss. See <a href=":url">https://getbootstrap.com/docs/5.1/customize/color/</a>', [':url' => 'https://getbootstrap.com/docs/5.1/customize/color/']),
      ],
    ];

    $form['components']['pager'] = [
      '#type' => 'details',
      '#title' => $this->t('Pager'),
      'pager_size' => [
        '#type' => 'select',
        '#title' => $this->t('Pager Size'),
        '#options' => [
          'default' => $this->t('Default'),
          'pagination-lg' => $this->t('Large'),
          'pagination-sm' => $this->t('Small'),
        ],
        '#default_value' => theme_get_setting('components.pager.pager_size', $theme_name),
        '#description' => $this->t('Select the size of the pager items.'),
      ],
      'pager_align' => [
        '#type' => 'select',
        '#title' => $this->t('Pager Alignment'),
        '#options' => [
          'default' => $this->t('Left (default)'),
          'justify-content-center' => $this->t('Center'),
          'justify-content-end' => $this->t('Right'),
        ],
        '#default_value' => theme_get_setting('components.pager.pager_align', $theme_name),
        '#description' => $this->t('Select the alignment for the pager items.'),
      ],
    ];
  }

  /**
   * Return a list of available color classes.
   *
   * @param string $prefix
   *   The class prefix. eg. 'btn'. Default is 'bg'.
   *
   * @return mixed
   *   An array of bootstrap color class options.
   */
  private function getColorClassOptions(string $prefix = 'bg') {
    return [
      'none' => $this->t('None'),
      $prefix . '-primary' => $this->t('Primary'),
      $prefix . '-secondary' => $this->t('Secondary'),
      $prefix . '-success' => $this->t('Success'),
      $prefix . '-info' => $this->t('Info'),
      $prefix . '-warning' => $this->t('Warning'),
      $prefix . '-danger' => $this->t('Danger'),
      $prefix . '-light' => $this->t('Light'),
      $prefix . '-dark' => $this->t('Dark'),
    ];
  }

}
