<?php

namespace Drush\Commands\monarch_config_zip;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Serialization\Yaml;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A Drush commandfile.
 */
final class MonarchConfigZipCommands extends DrushCommands {

  /**
   * Constructs a MonarchConfigZipCommands object.
   */
  public function __construct(
    private readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
    );
  }

  /**
   * Build a zip with all site configuration in YAML files.
   */
  private function buildConfigZip(): string {
    $zip_name = tempnam(sys_get_temp_dir(), 'config_zip_');

    $zip = new \ZipArchive();
    $zip->open($zip_name, \ZipArchive::CREATE | \ZipArchive::OVERWRITE);

    foreach ($this->configFactory->listAll() as $config_name) {
      $config = $this->configFactory->get($config_name)->getRawData();
      $yaml = Yaml::encode($config);
      $zip->addFromString($config_name . '.yml', $yaml);
    }

    $zip->close();

    return $zip_name;
  }

  /**
   * Dump site config as a zip file directly to stdout for piping via SSH.
   */
  #[CLI\Command(name: 'config:zip', aliases: ['cz'])]
  public function commandName(): void {
    // Don't try to view this interactively.
    $this->input()->setInteractive(FALSE);
    $file_name = $this->buildConfigZip();
    $file_data = file_get_contents($file_name);
    unlink($file_name);
    $this->output()->write($file_data, FALSE, OutputInterface::OUTPUT_RAW);
  }

}
