<?php

namespace Drush\Commands\monarch_config_zip;

use Consolidation\SiteAlias\SiteAliasManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Serialization\Yaml;
use Drupal\Core\Site\Settings;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * A Drush commandfile.
 */
final class MonarchConfigZipCommands extends DrushCommands {

  use AutowireTrait;

  /**
   * Constructs a MonarchConfigZipCommands object.
   */
  public function __construct(
    private readonly Settings $settings,
    private readonly SiteAliasManagerInterface $siteAliasManager,
    private readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct();
  }

  /**
   * Build a zip with all site configuration in YAML files.
   */
  private function buildConfigZip(): string {
    $zip_name = tempnam(sys_get_temp_dir(), 'config_zip_');

    $zip = new \ZipArchive();
    $zip->open($zip_name, \ZipArchive::CREATE | \ZipArchive::OVERWRITE);

    foreach ($this->configFactory->listAll() as $config_name) {
      $config = $this->configFactory->get($config_name)->getRawData();
      $yaml = Yaml::encode($config);
      $zip->addFromString($config_name . '.yml', $yaml);
    }

    $zip->close();

    return $zip_name;
  }

  /**
   * Dump site config as a zip file directly to stdout for piping via SSH.
   */
  #[CLI\Command(name: 'config:zip', aliases: ['cz'])]
  #[CLI\Usage(name: 'drush config:zip > config.zip', description: 'Export all site configuration to a zip file.')]
  #[CLI\Usage(name: 'drush @remote.dev cz > config.zip', description: 'Export configuration from a remote site to a local zip file.')]
  public function configZip(): void {
    // Don't try to view this interactively.
    $this->input()->setInteractive(FALSE);
    $file_name = $this->buildConfigZip();
    $file_data = file_get_contents($file_name);
    unlink($file_name);
    $this->output()->write($file_data, FALSE, OutputInterface::OUTPUT_RAW);
  }

  /**
   * Zip, download, and update config sync directory from specified drush alias.
   */
  #[CLI\Command(name: 'config:zip-pull', aliases: ['czp'])]
  #[CLI\Argument(name: 'source', description: 'The name of the remote Drush alias to pull configuration from.')]
  #[CLI\Usage(name: 'drush config:zip-pull @remote.dev', description: 'Pull configuration from the remote.dev alias and extract to the config sync directory.')]
  #[CLI\Usage(name: 'drush czp @remote.prod', description: 'Pull configuration from the remote.prod alias using the command alias.')]
  public function configZipPull(string $source): void {
    $config_sync_directory = $this->settings->get('config_sync_directory');

    if (empty($config_sync_directory)) {
      $this->logger()->error(dt('Config sync directory is not configured in settings.'));
      return;
    }

    $config_sync_directory_real = realpath($config_sync_directory);

    if ($config_sync_directory_real === FALSE) {
      $this->logger()->error(dt('Config sync directory does not exist: :dir', [':dir' => $config_sync_directory]));
      return;
    }

    $git_exists_process = $this->processManager()->shell('git --version', $config_sync_directory_real);
    $git_exists_process->mustRun();
    $git_exists = $git_exists_process->isSuccessful();

    if ($git_exists) {
      $git_diff_process = $this->processManager()->shell('git diff ' . escapeshellarg(rtrim($config_sync_directory_real, '/') . '/*.yml'));
      $git_diff_process->mustRun();

      if ($git_diff_process->isSuccessful()) {
        $diffs = trim($git_diff_process->getOutput());

        if (!empty($diffs)) {
          $this->logger()->error(dt('Uncommitted changes exist in the config sync directory (:dir). Please commit or stash these changes before pulling new configuration.', [':dir' => $config_sync_directory]));
          return;
        }
      }
      else {
        $this->logger()->error(dt('Failed to check for uncommitted changes in the config sync directory (:dir).', [':dir' => $config_sync_directory]));
        return;
      }
    }

    $sourceRecord = $this->siteAliasManager->get($source);

    if ($sourceRecord === NULL) {
      $this->logger()->error(dt('Site alias not found: :source', [':source' => $source]));
      return;
    }

    $this->logger()->notice(dt('Pulling zipped configuration from: :source', [':source' => $source]));
    $process = $this->processManager()->drush($sourceRecord, 'config:zip', []);
    $process->mustRun();

    $output = $process->getOutput();

    if (empty($output)) {
      $this->logger()->error(dt('No configuration data received from remote.'));
      return;
    }

    $temp_zip = tempnam(sys_get_temp_dir(), 'config_zip_pull_');

    if ($temp_zip === FALSE) {
      $this->logger()->error(dt('Failed to create temporary file for zip download.'));
      return;
    }

    if (file_put_contents($temp_zip, $output) === FALSE) {
      $this->logger()->error(dt('Failed to write zip data to temporary file.'));
      unlink($temp_zip);
      return;
    }

    $zip = new \ZipArchive();
    $zip_result = $zip->open($temp_zip);

    if ($zip_result !== TRUE) {
      $this->logger()->error(dt('Failed to open the downloaded zip file. Error code: :code', [':code' => $zip_result]));
      unlink($temp_zip);
      return;
    }

    $this->logger()->notice(dt('Extracting configuration to: :dir', [':dir' => $config_sync_directory]));
    $files = glob($config_sync_directory_real . '/*.yml');

    if ($files !== FALSE) {
      foreach ($files as $file) {
        if (!unlink($file)) {
          $this->logger()->warning(dt('Failed to delete existing config file: :file', [':file' => $file]));
        }
      }
    }

    if (!$zip->extractTo($config_sync_directory_real)) {
      $this->logger()->error(dt('Failed to extract configuration files to: :dir', [':dir' => $config_sync_directory_real]));
      $zip->close();
      unlink($temp_zip);
      return;
    }

    $zip->close();
    unlink($temp_zip);

    $this->logger()->notice(dt('Configuration zip and pull complete.'));
  }

}
