<?php

namespace Drupal\monarch_file_delete\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for uploading CSV files for file deletion operations.
 */
class DeleteFilesForm extends FormBase {

  const FORMATS = 'csv';

  /**
   * The file_system service.
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->fileSystem = $container->get('file_system');
    $instance->database = $container->get('database');
    $instance->messenger = $container->get('messenger');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'monarch_file_delete_files_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $description = <<<HTML
      <p>Upload a CSV of file names for bulk deletion.</p>
      <p>Files managed by Drupal will not be deleted.</p>
      <p>Filenames may be matched with or without the file extension. Matches are case insensitive, but otherwise must match exactly.</p>
      <p>Example: "agreedorder" matches "AgreedOrder"</p>
      <p>Example: "agreedorder" will not match "agreed order"</p>
    HTML;

    $form['description'] = [
      '#type' => 'markup',
      '#markup' => $description,
    ];

    $form['file'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Upload File'),
      '#description' => $this->t('Select a CSV file to upload.'),
      '#multiple' => FALSE,
      '#upload_validators' => [
        'FileExtension' => [
          'extensions' => self::FORMATS,
        ],
      ],
      '#upload_location' => 'temporary://file_delete_upload/',
      '#required' => TRUE,
    ];

    $form['header_row'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('CSV Has Header Row'),
      '#default_value' => TRUE,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Process File'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // File extensions are validated by the built-in file upload validator.
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $file = current($form_state->getValue('file') ?? []);
    $file = File::load($file);
    $file_uri = $file->getFileUri();
    $data = [];
    $found = [];

    $handle = fopen($file_uri, 'r');

    $header = $form_state->getValue('header_row') ? fgetcsv($handle) : [];

    while (($row = fgetcsv($handle)) !== FALSE) {
      foreach ($row as $value) {
        if ($value) {
          $data[] = $value;
        }
      }
    }

    $data = array_combine($data, $data);

    fclose($handle);

    $search_paths = [
      'public://',
      'private://',
      'temporary://',
    ];

    foreach ($search_paths as $search_path) {
      $search_path_real = $this->fileSystem->realpath($search_path);

      foreach ($data as $partial_filename) {
        $public_arg = escapeshellarg($search_path_real);
        $partial_filename_arg = escapeshellarg($partial_filename);
        $cmd = "find $public_arg -iname $partial_filename_arg'.*'; find $public_arg -iname $partial_filename_arg";
        $ret = `$cmd`;
  
        if ($ret) {
          $found_files = array_filter(array_map('trim', explode("\n", $ret)));
  
          foreach ($found_files as $found_file) {
            $found_file_uri = $search_path . ltrim(substr($found_file, strlen($search_path_real)), '/');
            $found[$found_file_uri] = $partial_filename;
          }
        }
      }
    }

    $success_messages = [];

    foreach ($found as $found_file_uri => $partial_filename) {
      unset($data[$partial_filename]);

      $fids = $this->database->select('file_managed', 'f')
        ->fields('f', ['fid'])
        ->condition('uri', $this->database->escapeLike($found_file_uri), 'LIKE')
        ->execute()
        ->fetchCol();

      if ($fids) {
        $this->messenger()->addError("Drupal managed file matched but not deleted: $found_file_uri");
        continue;
      }

      $this->fileSystem->unlink($found_file_uri);
      $success_messages[] = "File matched \"$partial_filename\" and deleted: $found_file_uri";
    }

    foreach ($data as $not_found_filename) {
      $this->messenger()->addWarning("File not found in public, private, or temporary file system: $not_found_filename");
    }

    foreach ($success_messages as $success_message) { 
      $this->messenger()->addMessage($success_message);
    }

    $this->messenger()->addMessage($this->t('CSV processed.'));
  }

}
