<?php

namespace Drupal\monarch_helper;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ModuleInstallerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Extension\ThemeInstallerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Update\UpdateHookRegistry;
use Drupal\mysql\Driver\Database\mysql\Connection;
use Drupal\user\PermissionHandler;
use Drupal\user\Entity\Role;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\Entity\FieldConfig;

/**
 * This helps with common update tasks.
 *
 * @todo Add More permissions helper functions as needed.
 * @todo Add node/entity helper functions as needed.
 */
final class ModuleUpdateHelperService {

  /**
   * An array for tracking log messages. (For duplicates?)
   */
  private array $logs = [];

  /**
   * Constructs a ModuleUpdateHelperService object.
   */
  public function __construct(
    private readonly ModuleInstallerInterface $moduleInstaller,
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly ThemeInstallerInterface $themeInstaller,
    private readonly ThemeHandlerInterface $themeHandler,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly Connection $database,
    private readonly LoggerChannelFactoryInterface $logger,
    private readonly UpdateHookRegistry $moduleUpdater,
    private readonly PermissionHandler $permissionsHandler,
  ) {}

  /**
   * Return module helper service.
   */
  public static function getService(): ModuleUpdateHelperService {
    return(\Drupal::service('module_update_helper.service'));
  }

  /**
   * Return another service.
   *
   * This is just a service return wrapper placeholder
   * just incase testing is implemented in the future.
   */
  public static function getServiceOther(string $service_name): mixed {
    return(\Drupal::service($service_name));
  }

  /**
   * Install one or more modules.
   */
  public function moduleInstall(string ...$modules): array {
    $r = [];
    foreach ($modules as $module) {
      if (!$this->moduleHandler->moduleExists($module)) {
        // Install one at a time so errors only stop the single function.
        $r[$module] = $this->moduleInstaller->install([$module]);
      }
    }
    $this->log('Installed Modules', $r);
    return $r;
  }

  /**
   * Uninstall one or more modules.
   */
  public function moduleUninstall(string ...$modules): array {
    $r = [];
    foreach ($modules as $module) {
      if ($this->moduleHandler->moduleExists($module)) {
        // Install one at a time so errors only stop the single function.
        $r[$module] = $this->moduleInstaller->uninstall([$module]);
      }
    }
    $this->log('Uninstalled Modules', $r);
    return $r;
  }

  /**
   * Set a module to a specified version.
   *
   * Can be used to make a module re-run its update hook.
   */
  public function moduleSchemaVersionSet(string $module, int $version = 8000): UpdateHookRegistry {
    $this->log('Setting module schema version', ['module' => $module, 'version' => $version]);
    return($this->moduleUpdater->setInstalledVersion($module, $version));
  }

  /**
   * Get the modules installed version.
   */
  public function moduleSchemaVersionGet(string $module): int {
    return($this->moduleUpdater->getInstalledVersion($module));
  }

  /**
   * Delete the modules installed version.
   */
  public function moduleSchemaVersionDelete(string $module): void {
    $this->log('Deleting module schema', $module);
    $this->moduleUpdater->deleteInstalledVersion($module);
  }

  /**
   * Install one or more themes.
   */
  public function themeInstall(string ...$themes): array {
    $r = [];
    foreach ($themes as $theme) {
      if (!$this->themeHandler->themeExists($theme)) {
        // Install one at a time so errors only stop the single function.
        $r[$theme] = $this->themeInstaller->install([$theme]);
      }
    }
    $this->log('Installed Themes', $r);
    return $r;
  }

  /**
   * Uninstall one or more modules.
   */
  public function themeUninstall(string ...$themes): array {
    $r = [];
    foreach ($themes as $theme) {
      if ($this->themeHandler->themeExists($theme)) {
        // Install one at a time so errors only stop the single function.
        $r[$theme] = $this->themeInstaller->uninstall([$theme]);
      }
    }
    $this->log('Uninstalled Themes', $r);
    return $r;
  }

  /**
   * Flus drupal caches if able.
   */
  public function cacheFlush(): bool {
    $r = FALSE;
    $kernel = NULL;
    if (function_exists('drupal_flush_all_caches')) {
      drupal_flush_all_caches($kernel);
      $r = TRUE;
    }
    $this->log('flushed caches', $r);
    return $r;
  }

  /**
   * Import a single config file.
   */
  public function configImportFile(string $file = '', string $config_name = '', string $method = 'merge'):Config|FALSE {
    // Get the config Object.
    $config = $this->configLoad($config_name);

    // Get the new config file.
    $fileContent = file_get_contents($file);
    if (!$fileContent) {
      $this->log('Unable to find YAML file', $file);
      return(FALSE);
    }

    // Decode the new config file.
    $yaml = Yaml::decode($fileContent);
    if (!$yaml) {
      $this->log('Unable to decode YAML file', $file);
      return(FALSE);
    }

    // Return the updated config.
    return($this->configUpdate($config_name, $method, $yaml));
  }

  /**
   * Update the config.
   */
  public function configUpdate(string $config_name, string $method = 'merge', array $data = [], bool $force = FALSE): Config|FALSE {
    if (empty($data)) {
      $this->log('Config Data is empty. If you are trying to delete config, please use the "configDelete" function. Use the "configLoad" function to get a config object.', $config_name);
      return(FALSE);
    }
    $config = $this->configLoad($config_name);
    // Get the config data as an array.
    $configData = $config->getRawData();
    if (empty($configData)) {
      $this->log('Unable to load config data', $config_name);
      if (!$force) {
        return(FALSE);
      }
    }

    switch ($method) {
      // Set all the data to the new keys and remove keys if able.
      case 'override':
        $config->setData($data);
        break;

      // Override existing keys but do not remove keys.
      case 'overlay':
        $newData = $configData;
        foreach ($data as $key => $value) {
          $newData[$key] = $value;
        }
        $config->setData($newData);
        break;

      // Add new keys only.
      case 'underlay':
        $newData = $configData;
        foreach ($data as $key => $value) {
          if (!isset($newData[$key])) {
            $newData[$key] = $value;
          }
        }
        $config->setData($newData);
        break;

      // Combine both config replacing where possible with new config.
      case 'merge':
      default:
        $config->setData(array_merge($configData, $data));
        break;
    }

    $this->log('Config Updated', ['config' => $config_name, 'method' => $method]);
    return($config->save());

  }

  /**
   * Load a config by name.
   */
  public function configLoad(string $config_name = ''): Config|FALSE {
    // Get the current config object.
    $config = $this->configFactory->getEditable($config_name);
    if (empty($config)) {
      $this->log('Unable to load config', $config_name);
      return(FALSE);
    }
    return($config);
  }

  /**
   * Delete a config.
   */
  public function configDelete(string $config_name): Config|FALSE {
    $config = $this->configLoad($config_name);
    $this->log('Deleting config', $config_name);
    return($config->delete()->save());
  }

  /**
   * Log a message.
   */
  public function log(string $message = '', mixed $data = NULL, string $level = 'notice') {
    $logData = [
      'message' => $message,
      'level' => $level,
      'data' => $data,
      'time' => microtime(),
    ];
    if (!in_array($level, ['notice', 'warning', 'error'])) {
      $logData['error'] = 'invalid $level = ' . strval($level);
      $this->logs[] = $logData;
      return($logData);
    }
    $logger = $this->logger->get('monarch_update_helper');
    if (!$logData) {
      $logData['error'] = 'Unable to load logger';
      $this->logs[] = $logData;
      return($logData);
    }
    $logData['sucsess'] = $logger->{$level}(json_encode($logData));

    $this->logs[] = $logData;
    return($logData);
  }

  /**
   * Get all of the current log messages.
   */
  public function logs(): array {
    return ($this->logs);
  }

  /**
   * Create a sandbox for batch processing update hooks.
   */
  public function batchInit(array $sandbox = []): array {
    $this->batchCalculate($sandbox);
    return ($sandbox);
  }

  /**
   * Get the items that need to be processed on the current batch.
   */
  public function batchItemsGet(array &$sandbox = []) {
    $length = max($sandbox['total'], $sandbox['current'] + $sandbox['items_per_batch']);
    $items = array_slice($sandbox['items'], $sandbox['current'], $length, TRUE);
    return($items);
  }

  /**
   * Calculate batch vars.
   */
  public function batchCalculate(array &$sandbox = []): void {
    $sandbox['items'] = $sandbox['items'] ?? [];
    $sandbox['total'] = $sandbox['total'] ?? count($sandbox['items']);
    $sandbox['current'] = $sandbox['current'] ?? 0;
    $sandbox['items_per_batch'] = $sandbox['items_per_batch'] ?? 10;
    $sandbox['progress'] = (empty($sandbox['current']) ? 1 : $sandbox['current']) / (empty($sandbox['total']) ? 100000000 : $sandbox['total']);
    $sandbox['#finished'] = ($sandbox['progress'] == 1);
    $this->log('Batch Info', $sandbox);
  }

  /**
   * Add the items that need to be processed on the current batch.
   */
  public function batchItemsAdd(array &$sandbox = [], mixed ...$items): void {
    $sandbox['items'] += $items;
    $this->batchCalculate($sandbox);
  }

  /**
   * Update a sandbox for batch processing update hooks.
   */
  public function batchUpdate(array &$sandbox = []): void {
    $length = max($sandbox['total'], $sandbox['current'] + $sandbox['items_per_batch']);
    $sandbox['current'] += $length;
    $this->batchCalculate($sandbox);
  }

  /**
   * Remove invalid permissions.
   */
  public function userRoleRemoveNonExistentPermissions(): array {
    $r = [];
    /** @var \Drupal\user\RoleInterface[] $roles */
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $permissions = array_keys($this->permissionsHandler->getPermissions());
    foreach ($roles as $role) {
      $role_permissions = $role->getPermissions();
      $differences = array_diff($role_permissions, $permissions);
      if ($differences) {
        foreach ($differences as $permission) {
          $r[$role->label() ?? ''][$permission] = FALSE;
          $role->revokePermission($permission);
        }
        $role->save();
      }
    }
    $this->log('Removing non-existent permissions from roles', $r);
    return($r);
  }

  /**
   * Change a role from one to another and return a list of affected users.
   */
  public function userRoleReplacePermission(string $remove_role, string $add_role, bool $delete_removed_role = TRUE): array {
    $affected_users = [];

    /**
     * @var UserStorage $usersHavingRole
     *
     * Get all the users with the role to replace.
     */
    $usersHavingRole = $this->entityTypeManager->getStorage('user')->loadByProperties(['roles' => $remove_role]);

    /* Replace the role on every user that has it */
    foreach ($usersHavingRole as $user) {
      $affected_users[$user->id] = [
        'remove' => $user->removeRole($remove_role),
        'add' => $user->addRole($add_role),
      ];
      $user->save();
    }

    /* Delete the removed role. */
    if ($delete_removed_role) {
      $role = Role::load($remove_role);
      if (!empty($role)) {
        $role->delete();
      }
      else {
        $this->log('Role cannot be found', ['role' => $remove_role]);
      }
    }

    return ($affected_users);
  }

  /**
   * Delete the field from an entity and return the affected fields.
   */
  public function entityFieldDelete(string $field_name, string|array $entityMachineNames = [], bool $delete_removed_field = TRUE): array {

    /**
     * @var \Drupal\Core\Entity\EntityTypeInterface[] $entity_types
     */
    $entity_types = \Drupal::entityTypeManager()->getDefinitions();
    $entityMachineNames = (is_string($entityMachineNames) ? [$entityMachineNames] : $entityMachineNames);
    $entity_machine_names = empty($entity_machine_names ?? []) ? array_keys($entity_types) : ($entity_machine_names ?? []);

    $fields = [];

    // Find the fields that need to be deleted.
    foreach ($entity_machine_names as $entity_machine_name) {
      if (isset($entity_types[$entity_machine_name])) {

        /**
         * @var \Drupal\field\FieldStorageConfigInterface|null $field_storage
         */
        $field_storage = FieldStorageConfig::loadByName($entity_machine_name, $field_name);

        if ($field_storage) {
          $fields[$entity_machine_name] = ($fields[$entity_machine_name] ?? []) + [
            'machine_name' => $field_name,
            'entity_type' => $field_storage->getEntityTypeId(),
            'bundles' => $field_storage->getBundles(),
            'dependencies' => $field_storage->getDependencies(),
            'description' => $field_storage->getDescription(),
            'settings' => $field_storage->getSettings(),
          ];
        }
      }
      else {
        $this->log('Entity Type does not Exist.', $entity_machine_name);
      }
    }

    // Loop threw all the fields that were found.
    foreach ($fields as &$field) {
      foreach ($field['bundles'] as $bundle) {
        $fieldConfig = FieldConfig::loadByName($field['entity_type'], $bundle, $field_name);
        $field['status'] = $field['status'] ?? 'processing';

        // Delete the field from the bundle.
        if (!empty($fieldConfig)) {
          $fieldConfig->delete();
          $field['status'] = 'processed';
        }
      }

      // Delete the field after it is removed from each bundle.
      if ($delete_removed_field) {
        $field_storage = FieldStorageConfig::loadByName($field['entity_type'], $field_name);
        if (!empty($field_storage) && $field['status'] == 'processed') {
          $field_storage->delete();
          $field['status'] = 'deleted';
        }
      }
      else {
        $field['status'] = 'processed';
      }
    }

    return ($fields);
  }

}
