<?php

namespace Drupal\monarch_media_canonical_download\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\file\Plugin\Field\FieldType\FileFieldItemList;
use Drupal\media\MediaInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Returns responses for COS Media as File routes.
 */
class MediaCanonicalDownloadController extends ControllerBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();

    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->configFactory = $container->get('config.factory');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->currentUser = $container->get('current_user');

    return $instance;
  }

  /**
   * Builds the response.
   */
  public function build(?MediaInterface $media = NULL, ?Request $request = NULL) {
    $config = $this->configFactory->get('monarch_media_canonical_download.settings');
    $settings = $config->get($media->bundle());
    $enabled = $settings['enabled'] ?? FALSE;
    $overrides = $settings['overrides'] ?? [];
    $ret = [];

    foreach ($media->getFieldDefinitions() as $field_name => $definition) {
      if ($field_name != 'thumbnail' && !($definition instanceof BaseFieldDefinition) && ($field = $media->get($field_name)) instanceof FileFieldItemList) {
        $file = $field->entity ?? NULL;
        $uri = ($file->uri->value ?? NULL);

        if ($file) {
          $mime = $file->getMimeType();
          $ext = pathinfo($uri, PATHINFO_EXTENSION);

          if ($enabled xor (($ext && in_array($ext, $overrides)) || ($mime && in_array($mime, $overrides)))) {
            $ret = [
              'uri' => $uri,
              'status' => 200,
              'headers' => [
                'Content-Type' => $file->getMimeType(),
                'Content-Length' => $file->getSize(),
                'Cache-Control' => 'private',
              ],
            ];

            break;
          }
        }
      }
    }

    $this->moduleHandler()->alter('media_canonical_download', $ret, $media, $this->currentUser, $request);

    if ($ret instanceof Response) {
      return $ret;
    }

    if (!empty($ret) && !empty($ret['uri'] ?? NULL)) {
      if (!file_exists($ret['uri'])) {
        return new Response('Expected file does not exist.', 500);
      }

      if ($settings['add_extension'] ?? NULL) {
        $header_keys = array_map(function ($value) {
          return strtolower(trim($value));
        }, array_keys($ret['headers'] ?? []));

        if (!in_array('content-disposition', $header_keys)) {
          $mime = _monarch_media_canonical_download_extension_list()[pathinfo(basename($request->getRequestUri()), PATHINFO_EXTENSION)] ?? NULL;
          /* If the files seem local and drupal managed, then use the good mime type check. */
          if (
            str_starts_with($ret['uri'], 'private:') ||
            str_starts_with($ret['uri'], 'public:') ||
            str_starts_with($ret['uri'], 'temporary:')
          ) {
            $uri_mime = mime_content_type($ret['uri']);
          }
          else {
            $uri_mime = _monarch_media_canonical_download_extension_list()[pathinfo(basename($ret['uri']), PATHINFO_EXTENSION)] ?? NULL;
          }

          if ($mime !== $uri_mime) {
            $filename = implode('.', array_filter([
              basename($request->getRequestUri()),
              pathinfo(basename($ret['uri']), PATHINFO_EXTENSION) ?: current(array_keys(_monarch_media_canonical_download_extension_list($uri_mime))) ?: '',
            ]));

            $ret['headers']['Content-Disposition'] = 'inline; filename="' . urlencode($filename) . '"';
          }
        }
      }

      return new BinaryFileResponse($ret['uri'], $ret['status'] ?? 200, $ret['headers'] ?? []);
    }

    return $this->entityTypeManager->getViewBuilder('media')->view($media);
  }

}
