<?php

namespace Drupal\monarch_migration_d7\Plugin\MigrationMapper;

use Drupal\migrate\Row;
use Drupal\Core\File\FileSystemInterface;
use Drupal\migrate\MigrateSkipRowException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Maps fields and copies files.
 *
 * Add these to your settings.local.php and set them to the correct paths:
 *
 * $config['migrate_plus.migration_group.monarch_migration_d7']['shared_configuration']['constants']['d7_public'] = '/path/to/public/files';
 * $config['migrate_plus.migration_group.monarch_migration_d7']['shared_configuration']['constants']['d7_private'] = '/path/to/private/files';
 *
 * @MigrationMapper(
 *   id = "monarch_migration_d7_file",
 *   migration = "file",
 *   weight = -9999,
 * )
 */
class File extends MigrationMapperD7Base {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->fileSystem = $container->get('file_system');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function requirements() : ?array {
    return ['user'];
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {
    $uri_parts = explode('://', $row->getSourceProperty('uri') ?? '');

    if ($uri_parts[0] !== 'public' && $uri_parts[0] !== 'private') {
      throw new MigrateSkipRowException('Unknown uri scheme. Ignoring file.');
    }

    $row->setDestinationProperty('uid', $row->getSourceProperty('uid'));
    $row->setDestinationProperty('fid', $row->getSourceProperty('fid'));
    $row->setDestinationProperty('filename', $row->getSourceProperty('filename'));
    $row->setDestinationProperty('uri', $row->getSourceProperty('uri'));
    $row->setDestinationProperty('filemime', $row->getSourceProperty('filemime'));
    if ($row->hasSourceProperty('status')) {
      $row->setDestinationProperty('status', $status = $row->getSourceProperty('status'));
      $row->setDestinationProperty('moderation_state', $status == 1 ? 'published' : 'draft');
    }
    $row->setDestinationProperty('created', $row->getSourceProperty('timestamp'));
    $row->setDestinationProperty('changed', $row->getSourceProperty('timestamp'));

    foreach (($this->fieldInfo['file'] = $this->fieldInfo['file'] ?? static::getBundleFields('file')) as $field_name => $field_type) {
      if ($row->hasSourceProperty($field_name)) {
        $row->setSourceProperty($field_name, $this->processField($field_type, $row->getSourceProperty($field_name)));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postRowSave(Row $row) {
    $source_uri = $row->getSourceProperty('uri');
    $dest_uri = $row->getDestinationProperty('uri');

    foreach ([
      'public://' => ($row->getSourceProperty('constants')['d7_public'] ?? NULL),
      'private://' => ($row->getSourceProperty('constants')['d7_private'] ?? NULL),
    ] as $scheme => $path) {
      if ($path && substr($source_uri, 0, strlen($scheme)) === $scheme) {
        $path = rtrim($path, '/');
        $source_path = $path . '/' . substr($source_uri, strlen($scheme));

        if (
          file_exists($source_path) &&
          $this->fileSystem->prepareDirectory($this->fileSystem->dirname($dest_uri), FileSystemInterface::MODIFY_PERMISSIONS || FileSystemInterface::CREATE_DIRECTORY)
        ) {
          $this->fileSystem->copy($source_path, $dest_uri, FileSystemInterface::EXISTS_REPLACE);
        }
      }

    }
  }

}
