<?php

namespace Drupal\monarch_migration_d7\Plugin\MigrationMapper;

use Drupal\migrate\Row;
use Drupal\migrate\MigrateSkipRowException;

/**
 * Maps files to media entities.
 *
 * @MigrationMapper(
 *   id = "monarch_migration_d7_media",
 *   migration = "media",
 *   weight = -9999,
 * )
 */
class Media extends MigrationMapperD7Base {

  const BUNDLE_MAP = [
    '/^image\\//' => 'image',
    '/^video\\//' => 'video',
    '/^audio\\//' => 'audio',
  ];

  const DEFAULT_BUNDLE = 'document';

  const BUNDLE_FIELD_MAP = [
    'image' => 'field_media_image',
    'video' => 'field_media_video_file',
    'audio' => 'field_media_audio_file',
    'document' => 'field_media_document',
  ];

  /**
   * {@inheritdoc}
   */
  public function requirements() : ?array {
    return ['file', 'user'];
  }

  /**
   * Get field details for $fid.
   */
  public function fileFieldReferenceLookup($fid, $type = NULL) {
    if (!empty($fid)) {
      $results = [];

      foreach ([
        'file' => ['display', 'description'],
        'image' => ['width', 'height', 'alt', 'title'],
      ] as $field_type => $columns) {
        if (!is_null($type) && $field_type !== $type) {
          continue;
        }

        $query = NULL;

        if ($fields = $this->sourceDatabase->select('field_config', 'fc')->fields('fc', ['field_name'])->condition('type', $field_type)->execute()->fetchCol()) {
          $field_name = reset($fields);

          do {
            $subquery = $this->sourceDatabase->select('field_data_' . $field_name)->condition($field_name . '_fid', $fid);

            foreach ($columns as $column) {
              $subquery->addField('field_data_' . $field_name, $field_name . '_' . $column, $column);
            }

            if ($query) {
              $query->union($subquery);
            }
            else {
              $query = $subquery;
            }
          } while (($field_name = next($fields)));
        }

        if ($query) {
          $res = $query->execute()->fetchAll();

          if ($type) {
            return $res ?? [];
          }

          if ($res) {
            $results[$field_type] = $res;
          }
        }
      }

      return $results;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {
    $row->setDestinationProperty('mid', $row->getSourceProperty('fid'));
    $row->setDestinationProperty('vid', $row->getSourceProperty('fid'));
    $row->setDestinationProperty('uid', $row->getSourceProperty('uid'));
    if ($row->hasSourceProperty('status')) {
      $row->setDestinationProperty('status', $status = $row->getSourceProperty('status'));
      $row->setDestinationProperty('moderation_state', $status == 1 ? 'published' : 'draft');
    }
    $row->setDestinationProperty('created', $row->getSourceProperty('timestamp'));
    $row->setDestinationProperty('changed', $row->getSourceProperty('timestamp'));
    $dest_fid = $row->getSourceProperty('fid');
    $row->setDestinationProperty('bundle', $bundle = static::DEFAULT_BUNDLE);

    foreach (($this->fieldInfo['file'] = $this->fieldInfo['file'] ?? static::getBundleFields('file')) as $field_name => $field_type) {
      if ($row->hasSourceProperty($field_name)) {
        $row->setSourceProperty($field_name, $this->processField($field_type, $row->getSourceProperty($field_name)));
      }
    }

    if ($dest_fid) {
      $mime = $row->getSourceProperty('filemime') ?? NULL;

      if (!$mime) {
        $uri = $row->getSourceProperty('uri');
        $mime = mime_content_type($uri);
      }

      if ($mime) {
        foreach (static::BUNDLE_MAP as $regexp => $possible_bundle) {
          if (preg_match($regexp, $mime)) {
            $row->setDestinationProperty('bundle', $bundle = $possible_bundle);
            break;
          }
        }
      }

      $field_data = [
        'target_id' => $dest_fid,
      ] + ((array) (current($this->fileFieldReferenceLookup($dest_fid, $bundle === 'image' ? 'image' : 'file')) ?: []));

      $row->setDestinationProperty(static::BUNDLE_FIELD_MAP[$bundle], $field_data);
    }
    else {
      throw new MigrateSkipRowException('File entity does not exist for this media!');
    }

  }

}
