<?php

namespace Drupal\monarch_migration_d7\Plugin\MigrationMapper;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\monarch_migration\Plugin\MigrationMapper\MigrationMapperBase;
use Drupal\Core\Entity\Sql\SqlContentEntityStorage;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\Entity\BaseFieldOverride;

/**
 * The migration mapper plugin base class.
 */
abstract class MigrationMapperD7Base extends MigrationMapperBase {

  const LINK_URI_SCHEME = 'http://';
  const DATE_FORMAT = 'Y-m-d\\TH:i:s';
  const MEDIA_WYSIWYG_TOKEN_REGEX = '/\[\[\{.*?"type":"media".*?\}\]\]/s';

  /**
   * Field info cache.
   *
   * @var array
   */
  protected $fieldInfo = [];

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->dateFormatter = $container->get('date.formatter');
    $instance->renderer = $container->get('renderer');

    return $instance;
  }

  /**
   * Get block data.
   */
  protected function getBlockData(string $moddelta = NULL, int $bid = NULL, bool $return_bid_only = FALSE, string $theme = NULL) {
    if (!is_null($moddelta)) {
      $moddelta_parts = explode(':', $moddelta);

      if (count($moddelta_parts) !== 2) {
        return [];
      }
    }
    else {
      $moddelta_parts = [];
    }

    $query = $this->sourceDatabase->select('block', 'b');
    $query->leftJoin('block_custom', 'bc', 'b.module = \'block\' and b.delta = bc.bid');
    if ($return_bid_only) {
      $query->fields('b', ['bid']);
    }
    else {
      $query->fields('b')->fields('bc', ['body', 'info', 'format']);
      $query->addField('bc', 'bid', 'bcid');
    }

    if (!empty($moddelta_parts)) {
      $query->condition('b.module', $moddelta_parts[0]);
      $query->condition('b.delta', $moddelta_parts[1]);
    }

    if (!is_null($bid)) {
      $query->condition('b.bid', $bid);
    }

    if (!is_null($theme)) {
      $query->condition('b.theme', $theme);
    }

    if (!empty($moddelta_parts) || !is_null($bid)) {
      $query->range(0, 1);
    }

    $ret = ($return_bid_only ? $query->execute()->fetchCol() : $query->execute()->fetchAll(\PDO::FETCH_ASSOC)) ?: [];

    $ret = current($ret) ?: NULL;

    if (!$return_bid_only && $ret) {
      if ($ret && $ret['module'] === 'menu_block') {
        $variable_prefix_length = strlen('menu_block_' . $ret['delta'] . '_');

        foreach (static::getVariable('menu\\_block\\_' . $ret['delta'] . '\\_%') as $key => $value) {
          $ret['menu_block'][substr($key, $variable_prefix_length)] = $value;
        }
      }

      return $ret;
    }

    return $ret;
  }

  /**
   * Lookup migrated block id.
   */
  protected function lookupMigratedBlockId(string $moddelta = NULL, int $bid = NULL) {
    if (is_null($moddelta) && is_null($bid)) {
      return NULL;
    }

    $data = static::getBlockData($moddelta, $bid, FALSE);
    $ret = current(is_null($data['bcid']) ? [] : ($this->migrateLookup->lookup('custom_block', [$data['bcid']]) ?: []));

    return $ret === FALSE ? NULL : $ret;
  }

  /**
   * Get the fields for an entity type and bundle.
   */
  public function getBundleFields(string $entity_type_id, string $bundle = NULL, bool $verbose = FALSE) {
    $ret = [];

    $query = $this->sourceDatabase->select('field_config', 'fc');
    $query->innerJoin('field_config_instance', 'fci', 'fci.field_id = fc.id');
    $query->addField('fc', 'field_name');
    $query->addField('fc', 'type');
    $query->condition('entity_type', $entity_type_id);

    if (!is_null($bundle)) {
      $query->condition('bundle', $bundle);
    }

    if ($verbose) {
      $query->addField('fc', 'module');
      $query->addField('fc', 'data', 'storage_data');
      $query->addField('fc', 'cardinality');
      $query->addField('fc', 'translatable');
      $query->addField('fci', 'entity_type');
      $query->addField('fci', 'bundle');
      $query->addField('fc', 'data', 'instance_data');
    }

    $fields = $query->execute()->fetchAll();

    foreach ($fields as &$field) {
      if ($verbose) {
        if (!empty($field->data ?? NULL)) {
          $field->data = unserialize($field->data);
        }

        if (!empty($field->instance_data ?? NULL)) {
          $field->instance_data = unserialize($field->instance_data);
        }

        if (!empty($field->storage_data ?? NULL)) {
          $field->storage_data = unserialize($field->storage_data);
        }

        $ret[$field->field_name] = $field;
      }
      else {
        $ret[$field->field_name] = $field->type;
      }
    }

    return $ret;
  }

  /**
   * Process field values for d9.
   */
  public function processField(string $field_type, array $field_values = NULL) {
    if (!empty($field_values)) {
      switch ($field_type) {
        case 'phone':
        case 'blockreference':
        case 'list_text':
        case 'list_boolean':
          break;

        case 'text':
        case 'text_long':
        case 'text_with_summary':
          foreach ($field_values as $delta => $values) {
            $field_values[$delta] = [
              'value' => static::processText($values['value'] ?? ''),
              'summary' => $values['summary'] ?? NULL,
              'format' => $values['format'] ?? NULL,
            ];
          }
          break;

        case 'email':
          foreach ($field_values as $delta => $values) {
            $email = $values['email'] ?? NULL;
            unset($values['email']);
            $field_values[$delta] = [
              'value' => $email,
            ] + $values;
          }
          break;

        case 'image':
        case 'file':
          foreach ($field_values as $delta => $values) {
            $fid = $values['fid'];
            unset($values['fid']);
            $field_values[$delta] = ['target_id' => $fid] + $values;
          }
          break;

        case 'taxonomy_term_reference':
          foreach ($field_values as $delta => $values) {
            $field_values[$delta] = ['target_id' => $values['tid']];
          }
          break;

        case 'link_field':
          foreach ($field_values as $delta => $values) {
            $field_values[$delta] = static::processLinkField($values);
          }
          break;

        case 'paragraphs':
          foreach ($field_values as $delta => $values) {
            $field_values[$delta] = [
              'target_id' => $values['value'],
              'target_revision_id' => $values['revision_id'],
            ];
          }
          break;

        case 'datetime':
          foreach ($field_values as $delta => $values) {
            foreach ([
              'value' => 'value',
              'value2' => 'end_value',
            ] as $source_key => $dest_key) {
              if (isset($values[$source_key])) {
                $value = $values[$source_key];
                unset($values[$source_key]);
                $values[$dest_key] = $this->dateFormatter->format(strtotime($value), 'custom', static::DATE_FORMAT);
              }
            }

            $field_values[$delta] = $values;
          }
          break;

        default:
          throw new \Error('Unhandled field type:' . $field_type);
      }
    }

    return $field_values;
  }

  /**
   * Get the field data for a field on an "entity".
   */
  public function getFieldData(string $entity_type_id, string $field_name, ?string $entity_id, ?string $revision_id, ?string $bundle, string $language = 'und') {
    if (!is_null($revision_id)) {
      $query = $this->sourceDatabase->select('field_revision_' . $field_name, 'fd');
      $query->condition('revision_id', (int) $revision_id);
    }
    else {
      $query = $this->sourceDatabase->select('field_data_' . $field_name, 'fd');
    }

    $query->fields('fd');
    $query->condition('entity_type', $entity_type_id);
    $query->condition('deleted', 0);

    if (!is_null($entity_id)) {
      $query->condition('entity_id', (int) $entity_id);
    }

    if (!is_null($bundle)) {
      $query->condition('bundle', $bundle);
    }

    $query->condition('language', $language);
    $query->orderBy('delta', 'ASC');

    $ret = [];

    $rows = $query->execute()->fetchAll();

    foreach ($rows as $row) {
      $delta = $row->delta;

      foreach ($row as $column => $value) {
        if (substr($column, 0, strlen($field_name) + 1) === $field_name . '_') {
          $ret[$delta][substr($column, strlen($field_name) + 1)] = $value;
        }
      }
    }

    $this->fieldInfo[$entity_type_id] = $this->fieldInfo[$entity_type_id] ?? static::getBundleFields($entity_type_id, NULL, FALSE);

    return static::processField($this->fieldInfo[$entity_type_id][$field_name], $ret);
  }

  /**
   * Get the all field data for fields on an "entity".
   */
  public function getAllFieldData(string $entity_type_id, string $entity_id, ?string $revision_id, ?string $bundle, string $language = 'und') {
    $ret = [];
    $fields = static::getBundleFields($entity_type_id, $bundle);

    foreach ($fields as $field_name => $field_type) {
      $ret[$field_name] = static::getFieldData($entity_type_id, $field_name, $entity_id, $revision_id, $bundle);
    }

    return $ret;
  }

  /**
   * Get bundle map for all fields.
   */
  public function getSourceConfiguredFieldMap(bool $verbose = FALSE) {
    $ret = [];

    $res = $this->sourceDatabase->select('field_config_instance', 'fci')
      ->fields('fci', ['entity_type', 'bundle'])
      ->distinct()
      ->execute();

    foreach ($res as $row) {
      $ret[$row->entity_type][$row->bundle] = static::getBundleFields($row->entity_type, $row->bundle, $verbose);
    }

    return $ret;
  }

  /**
   * {@inheritdoc}
   */
  public function sourceModuleEnabled(string $module_name) : bool {
    $query = $this->sourceDatabase->select('system', 's');
    $query->condition('name', $module_name);
    $query->condition('status', '1');
    return (int) $query->countQuery()->execute()->fetchField();
  }

  /**
   * Turn a Drupal 6/7 URI into a Drupal 8-compatible format.
   *
   * @param string $uri
   *   The 'url' value from Drupal 6/7.
   *
   * @return string
   *   The Drupal 8-compatible URI.
   */
  protected function canonicalizeLinkUri($uri) {
    // If the path starts with 2 slashes then it is always considered an
    // external URL without an explicit protocol part.
    // @todo Remove this when https://www.drupal.org/node/2744729 lands.
    if (strpos($uri, '//') === 0) {
      return static::LINK_URI_SCHEME . ltrim($uri, '/');
    }

    // If we already have a scheme, we're fine.
    if (parse_url($uri, PHP_URL_SCHEME)) {
      return $uri;
    }

    // Empty URI and non-links are allowed.
    if (empty($uri) || in_array($uri, ['<nolink>', '<none>'])) {
      return 'route:<nolink>';
    }

    // Remove the <front> component of the URL.
    if (strpos($uri, '<front>') === 0) {
      $uri = substr($uri, strlen('<front>'));
    }
    else {
      // List of unicode-encoded characters that were allowed in URLs,
      // according to link module in Drupal 7. Every character between &#x00BF;
      // and &#x00FF; (except × &#x00D7; and ÷ &#x00F7;) with the addition of
      // &#x0152;, &#x0153; and &#x0178;.
      // @see https://git.drupalcode.org/project/link/blob/7.x-1.5-beta2/link.module#L1382
      // cSpell:disable-next-line
      $link_i_chars = '¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõöøùúûüýþÿŒœŸ';

      // Pattern specific to internal links.
      $internal_pattern = "/^(?:[a-z0-9" . $link_i_chars . "_\-+\[\] ]+)";

      $directories = "(?:\/[a-z0-9" . $link_i_chars . "_\-\.~+%=&,$'#!():;*@\[\]]*)*";
      // Yes, four backslashes == a single backslash.
      $query = "(?:\/?\?([?a-z0-9" . $link_i_chars . "+_|\-\.~\/\\\\%=&,$'():;*@\[\]{} ]*))";
      $anchor = "(?:#[a-z0-9" . $link_i_chars . "_\-\.~+%=&,$'():;*@\[\]\/\?]*)";

      // The rest of the path for a standard URL.
      $end = $directories . '?' . $query . '?' . $anchor . '?$/i';

      if (!preg_match($internal_pattern . $end, $uri)) {
        $link_domains = '[a-z][a-z0-9-]{1,62}';

        // Starting a parenthesis group with (?: means that it is grouped,
        // but is not captured.
        $authentication = "(?:(?:(?:[\w\.\-\+!$&'\(\)*\+,;=" . $link_i_chars . "]|%[0-9a-f]{2})+(?::(?:[\w" . $link_i_chars . "\.\-\+%!$&'\(\)*\+,;=]|%[0-9a-f]{2})*)?)?@)";
        $domain = '(?:(?:[a-z0-9' . $link_i_chars . ']([a-z0-9' . $link_i_chars . '\-_\[\]])*)(\.(([a-z0-9' . $link_i_chars . '\-_\[\]])+\.)*(' . $link_domains . '|[a-z]{2}))?)';
        $ipv4 = '(?:[0-9]{1,3}(\.[0-9]{1,3}){3})';
        $ipv6 = '(?:[0-9a-fA-F]{1,4}(\:[0-9a-fA-F]{1,4}){7})';
        $port = '(?::([0-9]{1,5}))';

        // Pattern specific to external links.
        $external_pattern = '/^' . $authentication . '?(' . $domain . '|' . $ipv4 . '|' . $ipv6 . ' |localhost)' . $port . '?';
        if (preg_match($external_pattern . $end, $uri)) {
          return static::LINK_URI_SCHEME . $uri;
        }
      }
    }

    // Add the internal: scheme and ensure a leading slash.
    return 'internal:/' . ltrim($uri, '/');
  }

  /**
   * Fix a link value from D7.
   */
  public function processLinkField($value) {
    $attributes = unserialize($value['attributes']);
    // Drupal 6/7 link attributes might be double serialized.
    if (!is_array($attributes)) {
      $attributes = unserialize($attributes);
    }

    // In rare cases Drupal 6/7 link attributes are triple serialized. To avoid
    // further problems with them we set them to an empty array in this case.
    if (!is_array($attributes)) {
      $attributes = [];
    }

    // Massage the values into the correct form for the link.
    $route['uri'] = static::canonicalizeLinkUri($value['url']);
    $route['options']['attributes'] = $attributes;
    $route['title'] = $value['title'];
    return $route;
  }

  /**
   * Fix a text field value from D7.
   */
  public function processText($value) {
    if ($value && preg_match_all(static::MEDIA_WYSIWYG_TOKEN_REGEX, $value, $matches)) {
      foreach (array_unique($matches[0]) as $token) {
        $embed_data = json_decode(trim($token, '[]'));

        if ($embed_data->fid ?? NULL) {
          $mid = $this->migrateLookup->lookup(['media'], [$embed_data->fid])[0]['mid'] ?? NULL;
          if ($mid && ($media = $this->entityTypeManager->getStorage('media')->load($mid))) {
            $external_url = $embed_data->fields->external_url ?? NULL;
            $view_mode = $embed_data->view_mode ?? 'default';
            $attributes = $embed_data->attributes ?? ((object) []);

            $render = [
              '#type' => 'html_tag',
              '#tag' => 'drupal-media',
              '#attributes' => [
                'data-entity-type' => 'media',
                'data-align' => 'center',
                'data-view-mode' => $view_mode,
                'data-entity-uuid' => $media->uuid(),
              ] + ((array) $attributes),
            ];

            if ($external_url) {
              $render = [
                '#type' => 'html_tag',
                '#tag' => 'a',
                '#attributes' => [
                  'href' => $external_url,
                ],
                'media' => $render,
              ];
            }

            $new_token = (string) $this->renderer->renderRoot($render);
            $value = str_replace($token, $new_token, $value);
          }
        }
      }
    }

    return $value;
  }

  /**
   * Get variable(s).
   */
  public function getVariable(string $like_name) {
    $ret = [];

    foreach ($this->sourceDatabase->select('variable', 'v')
      ->condition('name', $like_name, 'LIKE')
      ->fields('v', ['name', 'value'])
      ->execute() as $row
    ) {
      if (is_string($row->value)) {
        $ret[$row->name] = unserialize($row->value);
      }
      else {
        $ret[$row->name] = $row->value;
      }
    }

    return $ret;
  }

  /**
   * Get aliases.
   */
  public function getAliases($alias_target) : array {
    if (!$this->sourceDatabase->schema()->tableExists('url_alias')) {
      return [];
    }

    return $this->sourceDatabase->select('url_alias', 'ua')->fields('ua', [
      'pid',
      'alias',
    ])->condition('source', $alias_target)->orderBy('pid', 'ASC')->execute()->fetchAllKeyed();
  }

  /**
   * Get redirects.
   */
  public function getRedirects($redirect_target) : array {
    if (!$this->sourceDatabase->schema()->tableExists('redirect')) {
      return [];
    }

    $ret = $this->sourceDatabase->select('redirect', 'r')->fields('r')->condition('redirect', $redirect_target)->orderBy('rid', 'ASC')->execute()->fetchAllAssoc('rid', \PDO::FETCH_ASSOC);

    foreach ($ret as &$row) {
      unset($row['rid']);

      if (is_string($row['source_options'] ?? NULL)) {
        $row['source_options'] = unserialize($row['source_options']);
      }
      else {
        $row['source_options'] = [];
      }

      if (is_string($row['redirect_options'] ?? NULL)) {
        $row['redirect_options'] = unserialize($row['redirect_options']);
      }
      else {
        $row['redirect_options'] = [];
      }

      // Check if the url begins with http.
      if (preg_match('#^http#', $row['redirect'])) {
        // Use it as is.
        $uri = $row['redirect'];
      }
      else {
        // Make the link internal.
        $uri = 'internal:/' . $row['redirect'];
      }

      // Check if there are options.
      if (!empty($row['redirect_options'])) {
        // Check if there is a query.
        $options = $row['redirect_options'];
        if (!empty($options['query'])) {
          // Add it to the end of the url.
          $uri .= '?' . http_build_query($options['query']);
        }
        if (!empty($options['fragment'])) {
          $uri .= '#' . $options['fragment'];
        }
      }

      $row['redirect'] = $uri;
    }

    return $ret;
  }

  /**
   * Get the destination plugin storage, if it's exposed.
   */
  protected function getDestinationStorage() : ?EntityStorageInterface {
    if (
      ($migration = $this->configuration['migration'] ?? NULL) &&
      ($destination = $migration->getDestinationPlugin()) &&
      (method_exists($destination, 'getStorage'))
    ) {
      return $destination->getStorage();
    }

    return NULL;
  }

  /**
   * Get revision ids for the specified primary id.
   */
  protected function getRevisionIds($id, array $exclude_revision_ids = NULL) : array {
    if (!($storage = $this->getDestinationStorage())) {
      return [];
    }

    $revision_table = $storage instanceof SqlContentEntityStorage ? $storage->getRevisionTable() : NULL;

    if (!$revision_table) {
      return NULL;
    }

    $entity_type = $storage->getEntityType();

    $id_key = $entity_type->getKey('id');
    $revision_key = $entity_type->getKey('revision');

    $query = $this->database->select($revision_table, 'r')->fields('r', [$revision_key])->condition($id_key, $id);

    if (!empty($exclude_revision_ids)) {
      $query->condition($revision_key, $exclude_revision_ids, 'NOT IN');
    }

    return $query->execute()->fetchAllKeyed(0, 0);
  }

  /**
   * Get the fields for an entity type and bundle.
   */
  public function getSourceFieldInstances(string $entity_type_id, string $bundle = NULL) : ?array {
    $ret = [];

    $query = $this->sourceDatabase->select('field_config', 'fc');
    $query->innerJoin('field_config_instance', 'fci', 'fci.field_id = fc.id');
    $query->addField('fc', 'field_name');
    $query->addField('fc', 'type');
    $query->condition('entity_type', $entity_type_id);
    $query->addField('fci', 'bundle');

    if ($bundle) {
      $query->condition('fci.bundle', $bundle);
    }

    $fields = $query->execute()->fetchAll();

    switch ($entity_type_id) {
      case 'node':
        $types = $this->sourceDatabase->select('node_type', 't')->fields('t', ['type'])->execute()->fetchCol() ?: [];

        foreach ($types as $type) {
          $ret[$type] = [];
        }

        break;

      case 'file':
        $types = $this->sourceDatabase->select('file_type', 't')->fields('t', ['type'])->execute()->fetchCol() ?: [];

        foreach ($types as $type) {
          $ret[$type] = [
            'fid' => 'fid',
            'uri' => 'uri',
          ];
        }

        break;

      case 'taxonomy_term':
        $types = $this->sourceDatabase->select('taxonomy_vocabulary', 't')->fields('t', ['machine_name'])->execute()->fetchCol() ?: [];

        foreach ($types as $type) {
          $ret[$type] = [];
        }

        break;

      case 'user':
        $ret['user'] = [
          'picture' => 'image',
        ];

        break;
    }

    foreach ($fields as &$field) {
      $ret[$field->bundle][$field->field_name] = $field->type;
    }

    if ($entity_type_id === 'file') {
      unset($ret['audio']['fid']);
      unset($ret['audio']['uri']);
      unset($ret['image']['fid']);
      unset($ret['image']['uri']);
      unset($ret['video']['fid']);
      unset($ret['video']['uri']);
    }

    if ($bundle) {
      return $ret[$bundle] ?? NULL;
    }

    return $ret;
  }

  /**
   * Get the fields for an entity type and bundle.
   */
  public function getDestFieldInstances(string $entity_type_id, string $bundle = NULL) : ?array {
    $ret = [];

    $bundle_entity_type = $this->entityTypeManager->getStorage($entity_type_id)->getEntityType()->getBundleEntityType();

    if (!$bundle_entity_type) {
      $bundle_ids = [$entity_type_id];
    }
    else {
      $bundle_ids = [];

      foreach ($this->entityTypeManager->getStorage($bundle_entity_type)->loadMultiple() as $bundle_definition) {
        $bundle_ids[] = $bundle_definition->id();
      }
    }

    foreach ($bundle_ids as $bundle_id) {
      if ($bundle && $bundle_id !== $bundle) {
        continue;
      }

      $ret[$bundle_id] = [];

      $ret[$bundle_id] = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);

      foreach ($ret[$bundle_id] as $field_name => $field_definition) {
        if ($field_definition instanceof BaseFieldDefinition || $field_definition instanceof BaseFieldOverride) {
          unset($ret[$bundle_id][$field_name]);
          continue;
        }

        $ret[$bundle_id][$field_name] = $field_definition->getType();
      }
    }

    if ($bundle) {
      return $ret[$bundle] ?? NULL;
    }

    return $ret;
  }

  /**
   * Gets a field value to reference an entity.
   */
  public function ref(EntityInterface $entity = NULL) {
    if (empty($entity)) {
      return NULL;
    }

    $ret = [];

    $ret['target_id'] = $entity->id();

    if ($entity instanceof RevisionableInterface) {
      $ret['target_revision_id'] = $entity->getRevisionId();
    }

    return $ret;
  }

  /**
   * Turns an array of entitites into a list of field values.
   */
  public function refs(array $entities = NULL) : ?array {
    return is_null($entities) ? NULL : array_map([$this, 'ref'], $entities);
  }

}
