<?php

namespace Drupal\monarch_migration_d7\Plugin\MigrationMapper;

use Drupal\migrate\Row;
use Drupal\migrate\MigrateSkipRowException;

/**
 * Maps fields and attempts to skip user 0 and user 1.
 *
 * @MigrationMapper(
 *   id = "monarch_migration_d7_user",
 *   migration = "user",
 *   weight = -9999,
 * )
 */
class User extends MigrationMapperD7Base {

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {
    $uid = $row->getSourceProperty('uid');

    if ($uid == 0 || $uid == 1) {
      throw new MigrateSkipRowException('User id belongs to the Site Admin. Do not migrate!');
    }

    $row->setDestinationProperty('uid', $uid);
    $row->setDestinationProperty('name', $row->getSourceProperty('name'));
    $row->setDestinationProperty('pass', $row->getSourceProperty('pass'));
    $row->setDestinationProperty('mail', $row->getSourceProperty('mail'));
    $row->setDestinationProperty('created', $row->getSourceProperty('created'));
    $row->setDestinationProperty('access', $row->getSourceProperty('access'));
    $row->setDestinationProperty('login', $row->getSourceProperty('login'));
    $row->setDestinationProperty('status', $row->getSourceProperty('status'));
    $row->setDestinationProperty('timezone', $row->getSourceProperty('timezone'));
    $row->setDestinationProperty('init', $row->getSourceProperty('init'));
    $picture_fid = $row->getSourceProperty('picture');
    $row->setSourceProperty('picture', $picture_fid ? [['target_id' => $picture_fid]] : NULL);

    $lookup_path = "user/$uid";
    $aliases = $this->getAliases($lookup_path);

    if ($alias = array_pop($aliases)) {
      $row->setDestinationProperty('path', [
        'alias' => '/' . ltrim($alias, '/'),
        'pathauto' => FALSE,
      ]);
    }

    foreach (($this->fieldInfo['user'] = $this->fieldInfo['user'] ?? static::getBundleFields('user')) as $field_name => $field_type) {
      if ($row->hasSourceProperty($field_name)) {
        $row->setSourceProperty($field_name, $this->processField($field_type, $row->getSourceProperty($field_name)));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function processRow(Row $row) {
    $row->setDestinationProperty('langcode', $this->executePlugin('user_langcode', $row->getSourceProperty('entity_language'), $row, [
      'fallback_to_site_default' => FALSE,
    ]));

    $preferred_lang = $this->executePlugin('user_langcode', $row->getSourceProperty('language'), $row, [
      'fallback_to_site_default' => FALSE,
    ]);

    $row->setDestinationProperty('preferred_langcode', $preferred_lang);
    $row->setDestinationProperty('preferred_admin_langcode', $preferred_lang);
  }

  /**
   * {@inheritdoc}
   */
  public function postRowSave(Row $row) {
    $uid = $row->getSourceProperty('uid');
    $lookup_path = "user/$uid";
    $aliases = $this->getAliases($lookup_path);
    $redirects = $this->getRedirects($lookup_path);
    array_pop($aliases);

    if (count($aliases) > 0) {
      foreach ($aliases as $pid => $alias) {
        $this->createRedirect("new_alias_redirect:$pid", [
          'uid' => 0,
          'language' => 'und',
          'status_code' => '301',
          'redirect_source' => [
            'path' => ltrim($alias, '/'),
            'query' => NULL,
          ],
          'redirect_redirect' => [
            'uri' => "internal:/$lookup_path",
          ],
        ]);
      }
    }

    if (count($redirects) > 0) {
      foreach ($redirects as $rid => $redirect) {
        $this->createRedirect("new_redirect:$rid", [
          'uid' => $redirect['uid'],
          'language' => $redirect['language'],
          'status_code' => $redirect['status_code'] ?: '307',
          'redirect_source' => [
            'path' => $redirect['source'],
            'query' => $redirect['source_options']['query'] ?? NULL,
          ],
          'redirect_redirect' => [
            'uri' => "internal:/$lookup_path",
          ],
        ]);
      }
    }
  }

}
