<?php

namespace Drupal\monarch_migration\EventSubscriber;

use Drupal\migrate\Event\MigrateEvents;
use Drupal\migrate_plus\Event\MigrateEvents as MigratePlusEvents;
use Drupal\migrate\Event\MigratePreRowSaveEvent;
use Drupal\migrate\Event\MigratePostRowSaveEvent;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\monarch_migration\MigrationMapperPluginManager;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\migrate_plus\Event\MigratePrepareRowEvent;

/**
 * The monarch_migration event subscriber.
 */
class MonarchMigrationSubscriber implements EventSubscriberInterface {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The migration mapper plugin manager.
   *
   * @var \Drupal\monarch_migration\MigrationMapperPluginManager
   */
  protected $pluginManager;

  /**
   * Set up the services.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\monarch_migration\MigrationMapperPluginManager $plugin_manager
   *   The migration mapper plugin manager.
   */
  public function __construct(ModuleHandlerInterface $module_handler, MigrationMapperPluginManager $plugin_manager) {
    $this->moduleHandler = $module_handler;
    $this->pluginManager = $plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      MigrateEvents::PRE_ROW_SAVE => ['preRowSave'],
      MigrateEvents::POST_ROW_SAVE => ['postRowSave'],
      MigratePlusEvents::PREPARE_ROW => ['prepareRow'],
    ];
  }

  /**
   * The event handler.
   */
  public function prepareRow(MigratePrepareRowEvent $event) {
    $migration = $event->getMigration();
    $row = $event->getRow();
    $event->isPrepared = $event->isPrepared ?? NULL;

    $row->setDestinationProperty('migration_source', $migration->getPluginId());

    if (!$event->isPrepared) {
      foreach ($this->pluginManager->getPlugins($migration->getPluginId(), [
        'migration' => $migration,
      ]) as $plugin) {
        /** @var \Drupal\monarch_migration\MigrationMapperInterface $plugin */
        $plugin->prepareRow($row);
      }

      $event->isPrepared = TRUE;
    }

    return NULL;
  }

  /**
   * The event handler.
   */
  public function preRowSave(MigratePreRowSaveEvent $event) {
    $event->getRow()->removeDestinationProperty('__monarch_migration_process_helper__');
  }

  /**
   * The event handler.
   */
  public function postRowSave(MigratePostRowSaveEvent $event) {
    $migration = $event->getMigration();
    $row = $event->getRow();

    foreach ($this->pluginManager->getPlugins($migration->getPluginId(), [
      'migration' => $migration,
    ]) as $plugin) {
      /** @var \Drupal\monarch_migration\MigrationMapperInterface $plugin */
      $plugin->postRowSave($row);
    }
  }

}
