<?php

namespace Drupal\monarch_migration\Plugin\MigrationMapper;

use Drupal\migrate\Row;
use Drupal\block\BlockInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\migrate\MigrateException;
use Drupal\paragraphs\ParagraphInterface;
use Drupal\migrate_tools\MigrateExecutable;
use Drupal\monarch_migration\MigrationMapperInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\redirect\Entity\Redirect;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The migration mapper plugin base class.
 */
abstract class MigrationMapperBase extends PluginBase implements MigrationMapperInterface, ContainerFactoryPluginInterface {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The current database.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The legacy database.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $sourceDatabase;

  /**
   * The migration plugin manager.
   *
   * @var \Drupal\migrate\Plugin\MigrationPluginManagerInterface
   */
  protected $migrationPluginManager;

  /**
   * The source plugin manager.
   *
   * @var \Drupal\migrate\Plugin\MigratePluginManagerInterface
   */
  protected $sourcePluginManager;

  /**
   * The process plugin manager.
   *
   * @var \Drupal\migrate\Plugin\MigratePluginManagerInterface
   */
  protected $processPluginManager;

  /**
   * The destination plugin manager.
   *
   * @var \Drupal\migrate\Plugin\MigratePluginManagerInterface
   */
  protected $destinationPluginManager;

  /**
   * The migrate.lookup service.
   *
   * @var \Drupal\migrate\MigrateLookupInterface
   */
  protected $migrateLookup;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition
    );

    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->database = $container->get('database');
    $instance->migrationPluginManager = $container->get('plugin.manager.migration');
    $instance->sourcePluginManager = $container->get('plugin.manager.migrate.source');
    $instance->processPluginManager = $container->get('plugin.manager.migrate.process');
    $instance->destinationPluginManager = $container->get('plugin.manager.migrate.destination');
    $instance->migrateLookup = $container->get('migrate.lookup');

    if (
      ($configuration['migration'] ?? NULL) &&
      method_exists($configuration['migration'], 'getSourcePlugin') &&
      ($source_plugin = $configuration['migration']->getSourcePlugin()) &&
      method_exists($source_plugin, 'getDatabase')
    ) {
      $instance->sourceDatabase = $source_plugin->getDatabase();
    }

    return $instance;
  }

  /**
   * Define migration requirements for this plugin.
   */
  public function requirements() : ?array {
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function definitions() : ?array {
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function definitionAlter(array &$definition) {
    $reqs = static::requirements();

    if ($reqs) {
      $definition['migration_dependencies']['required'] = $definition['migration_dependencies']['required'] ?? [];
      if (is_array($definition['migration_dependencies']['required'])) {
        foreach ($reqs as $req) {
          if (!in_array($req, $definition['migration_dependencies']['required'])) {
            $definition['migration_dependencies']['required'][] = $req;
          }
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {}

  /**
   * {@inheritdoc}
   */
  public function cleanup(Row $row, EntityInterface $entity) {}

  /**
   * {@inheritdoc}
   */
  public function postRowSave(Row $row) {}

  /**
   * {@inheritdoc}
   */
  public function processRow(Row $row) {}

  /**
   * {@inheritdoc}
   */
  public function getMigrations() : array {
    $ret = $this->pluginDefinition['migration'] ?? [];

    if (!is_array($ret)) {
      $ret = [$ret];
    }

    return array_filter($ret);
  }

  /**
   * The get the source module names.
   */
  public function getSourceModules() : array {
    $ret = $this->pluginDefinition['source_module'] ?? [];

    if (!is_array($ret)) {
      $ret = [$ret];
    }

    return array_filter($ret);
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(string $migration_id) {
    $migrations = static::getMigrations();
    $source_modules = static::getSourceModules();

    if (
      !empty($migrations) &&
      !in_array($migration_id, $migrations)
    ) {
      return FALSE;
    }

    if (!empty($source_modules)) {
      foreach ($source_modules as $source_module) {
        if (!static::sourceModuleEnabled($source_module)) {
          return FALSE;
        }
      }
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight() {
    $this->pluginDefinition['weight'] ?? 0;
  }

  /**
   * Execute a plugin during the processRow event.
   */
  public function executePlugin(string $plugin_name, $value, Row $row, array $configuration = []) {
    $migration = $this->configuration['migration'];
    $migrate_executable = $this->configuration['migrate_executable'];

    if ($migration && $migrate_executable) {
      /** @var \Drupal\migrate\Plugin\MigrateProcessInterface $plugin */
      $plugin = $this->processPluginManager->createInstance($plugin_name, $configuration, $this->configuration['migration']);
      return $plugin->transform($value, $migrate_executable, $row, $this->configuration['destination_property'] ?? '');
    }

    throw new MigrateException('Only allowed to use executePlugin calls within processRow!');
  }

  /**
   * {@inheritdoc}
   */
  public function sourceModuleEnabled(string $module_name) {
    throw new \Exception('sourceModuleEnabled not implemented in ' . get_class($this));
  }

  /**
   * Create arbitrary paragraph tracked by the 'create_paragraph' migration.
   */
  public function createParagraph(string $paragraph_instance_id, string $bundle, array $fields_data = [], array $paragraph_behavior_settings = []) : ?ParagraphInterface {
    try {
      /** @var \Drupal\migrate\Plugin\MigrationInterface $migration */
      $migration = $this->migrationPluginManager->createInstance('create_paragraph', [
        'source' => [
          'data_rows' => [
            [
              'instance_id' => $paragraph_instance_id,
              'type' => $bundle,
              'bundle' => $bundle,
              'behavior_settings' => serialize($paragraph_behavior_settings),
            ] + $fields_data,
          ],
        ],
      ]);

      $migration->getIdMap()->prepareUpdate();
      $migrate_executable = new MigrateExecutable($migration);
      $migrate_executable->import();
      $ret = $this->migrateLookup->lookup('create_paragraph', [$paragraph_instance_id]);

      if ($ret) {
        $ret = current($ret);

        if (!is_null($ret['revision_id'] ?? NULL)) {
          return $this->entityTypeManager->getStorage('paragraph')->loadRevision($ret['revision_id']) ?: NULL;
        }

        if (!is_null($ret['id'] ?? NULL)) {
          if ($entity = ($this->entityTypeManager->getStorage('paragraph')->load($ret['id']) ?: NULL)) {
            $entity->setSyncing(TRUE);
            $entity->setNewRevision(FALSE);
          }
          else {
            return NULL;
          }

          return $entity;
        }
      }
    }
    catch (\Throwable $e) {
      throw new MigrateException($e->getMessage(), $e->getCode(), $e);
    }

    return NULL;
  }

  /**
   * Create arbitrary block tracked by the 'create_block' migration.
   */
  public function createBlock(string $block_instance_id, array $config_data = []) : ?BlockInterface {
    try {
      /** @var \Drupal\migrate\Plugin\MigrationInterface $migration */
      $migration = $this->migrationPluginManager->createInstance('create_block', [
        'source' => [
          'data_rows' => [
            [
              'instance_id' => $block_instance_id,
            ] + $config_data,
          ],
        ],
      ]);

      $migration->getIdMap()->prepareUpdate();
      $migrate_executable = new MigrateExecutable($migration);
      $migrate_executable->import();
      $ret = $this->migrateLookup->lookup('create_block', [$block_instance_id]);

      if ($ret) {
        $ret = current($ret)['id'] ?? NULL;

        if (is_null($ret)) {
          return NULL;
        }

        if ($entity = ($this->entityTypeManager->getStorage('block')->load($ret) ?: NULL)) {
          $entity->setSyncing(TRUE);
        }
        else {
          return NULL;
        }

        return $entity;

      }
    }
    catch (\Throwable $e) {
      throw new MigrateException($e->getMessage(), $e->getCode(), $e);
    }

    return NULL;
  }

  /**
   * Create arbitrary redirect tracked by the 'create_redirect' migration.
   */
  public function createRedirect(string $redirect_instance_id, array $config_data = []) : ?Redirect {
    try {
      /** @var \Drupal\migrate\Plugin\MigrationInterface $migration */
      $migration = $this->migrationPluginManager->createInstance('create_redirect', [
        'source' => [
          'data_rows' => [
            [
              'instance_id' => $redirect_instance_id,
            ] + $config_data,
          ],
        ],
      ]);

      $migration->getIdMap()->prepareUpdate();
      $migrate_executable = new MigrateExecutable($migration);
      $migrate_executable->import();
      $ret = $this->migrateLookup->lookup('create_redirect', [$redirect_instance_id]);

      if ($ret) {
        $ret = current($ret)['rid'] ?? NULL;

        if (is_null($ret)) {
          return NULL;
        }

        if ($entity = ($this->entityTypeManager->getStorage('redirect')->load($ret) ?: NULL)) {
          $entity->setSyncing(TRUE);
        }
        else {
          return NULL;
        }

        return $entity;
      }
    }
    catch (\Throwable $e) {
      throw new MigrateException($e->getMessage(), $e->getCode(), $e);
    }

    return NULL;
  }

  /**
   * Modify to make a machine name.
   */
  public function machineName(string $original_string, string $delimiter = '_', int $max_length = 32) : string {
    if ($original_string) {
      $quoted_delimiter = preg_quote($delimiter);
      return substr(trim(preg_replace("/$quoted_delimiter?[^a-z0-9$quoted_delimiter]+$quoted_delimiter?/i", $delimiter, $original_string), $delimiter), 0, $max_length);
    }

    return $original_string;
  }

}
