<?php

namespace Drupal\monarch_migration\Plugin\migrate\destination;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\SynchronizableInterface;
use Drupal\Core\Field\FieldTypePluginManagerInterface;
use Drupal\Core\Session\AccountSwitcherInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\TypedDataInterface;
use Drupal\migrate\MigrateException;
use Drupal\migrate\Plugin\migrate\destination\EntityContentBase;
use Drupal\migrate\Plugin\MigrateIdMapInterface;
use Drupal\migrate\Plugin\MigrationInterface;
use Drupal\migrate\Row;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin for editing/cleaning entity revisions in place.
 *
 * @MigrateDestination(
 *   id = "entity_in_place",
 *   deriver = "Drupal\monarch_migration\Plugin\Derivative\MigrateEntityInPlace"
 * )
 */
class EntityInPlace extends EntityContentBase {

  /**
   * The database service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The migration mapper plugin manager.
   *
   * @var \Drupal\monarch_migration\MigrationMapperPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition, MigrationInterface $migration = NULL) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition, $migration);

    $instance->database = $container->get('database');
    $instance->pluginManager = $container->get('plugin.manager.monarch_migration.hook');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, MigrationInterface $migration, EntityStorageInterface $storage, array $bundles, EntityFieldManagerInterface $entity_field_manager, FieldTypePluginManagerInterface $field_type_manager, AccountSwitcherInterface $account_switcher) {
    $plugin_definition += [
      'label' => new TranslatableMarkup('@entity_type revisions', ['@entity_type' => $storage->getEntityType()->getSingularLabel()]),
    ];
    parent::__construct($configuration, $plugin_id, $plugin_definition, $migration, $storage, $bundles, $entity_field_manager, $field_type_manager, $account_switcher);
  }

  /**
   * {@inheritdoc}
   */
  protected function updateEntity(EntityInterface $entity, Row $row) {
    $empty_destinations = $row->getEmptyDestinationProperties();
    // By default, an update will be preserved.
    $rollback_action = MigrateIdMapInterface::ROLLBACK_PRESERVE;

    foreach ($row->getDestination() as $field_name => $values) {
      $field = $entity->$field_name ?? NULL;

      if ($field instanceof TypedDataInterface) {
        $field->setValue($values);
      }
    }
    foreach ($empty_destinations as $field_name) {
      $field = $entity->$field_name ?? NULL;

      if ($field instanceof TypedDataInterface) {
        $field->setValue(NULL);
      }
    }

    $this->setRollbackAction($row->getIdMap(), $rollback_action);

    return $entity;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEntity(Row $row, array $old_destination_id_values) {
    $id_key = $this->getKey('id');
    $id = reset($old_destination_id_values);
    $id = $row->getDestinationProperty($id_key) ?: $id;
    $revision_id = NULL;
    $is_revisionable = FALSE;
    $entity = NULL;

    if ($revision_key = $this->getKey('revision')) {
      $is_revisionable = TRUE;
      $revision_id = next($old_destination_id_values);
      $revision_id = $row->getDestinationProperty($revision_key) ?: $revision_id;

      if (
        $revision_id !== FALSE &&
        !is_null($revision_id) &&
        ($entity = $this->storage->loadRevision($revision_id))
      ) {
        if ($entity->id() != $id) {
          throw new \Error('Entity ID does not match expected ID.');
        }

        $entity->enforceIsNew(FALSE);

        if ($entity instanceof RevisionableInterface) {
          $entity->setNewRevision(FALSE);
        }
      }
    }

    if (
      !$entity &&
      $id !== FALSE &&
      !is_null($id) &&
      ($entity = $this->storage->load($id))
    ) {
      $entity->enforceIsNew(FALSE);

      if ($is_revisionable && $entity instanceof RevisionableInterface) {
        $entity->setNewRevision(TRUE);
      }
    }

    if (!$entity) {
      $values = $row->getDestination();
      $bundle_key = $this->getKey('bundle');
      $values[$bundle_key] = $values[$bundle_key] ?? NULL;
      $this->configuration['default_bundle'] = $this->configuration['default_bundle'] ?? NULL;
      $values[$bundle_key] = $values[$bundle_key] ?: $this->configuration['default_bundle'] ?: $this->storage->getEntityTypeId();
      $entity = $this->storage->create($values);
      $entity->enforceIsNew(TRUE);

      if ($is_revisionable && $entity instanceof RevisionableInterface) {
        $entity->setNewRevision(FALSE);
      }
    }

    if ($entity instanceof RevisionableInterface && $entity instanceof EntityPublishedInterface) {
      $entity->isDefaultRevision($entity->isPublished());
    }

    return $this->updateEntity($entity, $row);
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\migrate\MigrateException
   *   When an entity cannot be looked up.
   * @throws \Drupal\migrate\Exception\EntityValidationException
   *   When an entity validation hasn't been passed.
   */
  public function import(Row $row, array $old_destination_id_values = []) {
    $this->rollbackAction = MigrateIdMapInterface::ROLLBACK_DELETE;
    $entity = $this->getEntity($row, $old_destination_id_values);

    if (!$entity) {
      throw new MigrateException('Unable to get entity');
    }

    assert($entity instanceof ContentEntityInterface);

    if ($this->isEntityValidationRequired($entity)) {
      $this->validateEntity($entity);
    }

    $ids = $this->save($entity, $old_destination_id_values);

    foreach ($this->pluginManager->getPlugins($this->migration->getPluginId(), [
      'migration' => $this->migration,
    ]) as $plugin) {
      /** @var \Drupal\monarch_migration\MigrationMapperInterface $plugin */
      $plugin->cleanup($row, $entity);
    }

    return $ids;
  }

  /**
   * Get entity hash.
   */
  protected function getEntityDataHash(EntityInterface $entity, array $field_names = NULL) {
    $fields = $entity->toArray();

    if ($field_names) {
      foreach (array_keys($fields) as $field_name) {
        if (!in_array($field_name, $field_names)) {
          unset($fields[$field_name]);
        }
      }
    }
    else {
      foreach ($entity->getEntityType()->getKeys() as $key_name) {
        if ($key_name !== 'status' && $key_name !== 'owner') {
          unset($fields[$key_name]);
        }
      }

      foreach (array_keys($fields) as $field_name) {
        if (substr($field_name, 0, 9) === 'revision_') {
          unset($fields[$field_name]);
        }
      }

      unset($fields['created']);
      unset($fields['changed']);

      $fields = array_filter($fields);
    }

    ksort($fields);

    return md5(serialize($fields));
  }

  /**
   * {@inheritdoc}
   */
  protected function save(ContentEntityInterface $entity, array $old_destination_id_values = []) {
    if ($entity instanceof SynchronizableInterface) {
      $entity->setSyncing(TRUE);
    }

    $entity->save();

    $ids = $old_destination_id_values;

    foreach (array_keys($this->getIds()) as $index => $field_name) {
      $ids[$index] = $entity->{$field_name}->value;
    }

    return $ids;
  }

  /**
   * {@inheritdoc}
   */
  public function getIds() {
    $ids = [];

    if ($id_key = $this->getKey('id')) {
      $ids[$id_key] = $this->getDefinitionFromEntity($id_key);
    }

    if ($revision_key = $this->getKey('revision')) {
      $ids[$revision_key] = $this->getDefinitionFromEntity($revision_key);
    }

    if ($this->isTranslationDestination()) {
      if ($langcode_key = $this->getKey('langcode')) {
        $ids[$langcode_key] = $this->getDefinitionFromEntity($langcode_key);
      }
    }

    return $ids;
  }

  /**
   * {@inheritdoc}
   */
  public function getHighestId() {
    $values = $this->storage->getQuery()
      ->accessCheck(FALSE)
      ->allRevisions()
      ->sort($this->getKey('revision'), 'DESC')
      ->range(0, 1)
      ->execute();
    // The array keys are the revision IDs.
    // The array contains only one entry, so we can use key().
    return (int) key($values);
  }

  /**
   * Gets the storage associated with the plugin.
   */
  public function getStorage() : EntityStorageInterface {
    return $this->storage;
  }

}
