<?php

namespace Drupal\monarch_migration_d7\Form;

use Drupal\Core\Database\Database;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\Entity\BaseFieldOverride;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Monarch Migration D7 settings for this site.
 */
class FieldMap extends ConfigFormBase {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The current database.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The legacy database.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $sourceDatabase;

  /**
   * The migration plugin manager.
   *
   * @var \Drupal\migrate\Plugin\MigrationPluginManagerInterface
   */
  protected $migrationPluginManager;

  /**
   * The source plugin manager.
   *
   * @var \Drupal\migrate\Plugin\MigratePluginManagerInterface
   */
  protected $sourcePluginManager;

  /**
   * The process plugin manager.
   *
   * @var \Drupal\migrate\Plugin\MigratePluginManagerInterface
   */
  protected $processPluginManager;

  /**
   * The destination plugin manager.
   *
   * @var \Drupal\migrate\Plugin\MigratePluginManagerInterface
   */
  protected $destinationPluginManager;

  /**
   * The migrate.lookup service.
   *
   * @var \Drupal\migrate\MigrateLookupInterface
   */
  protected $migrateLookup;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->migrationPluginManager = $container->get('plugin.manager.migration');
    $instance->sourcePluginManager = $container->get('plugin.manager.migrate.source');
    $instance->processPluginManager = $container->get('plugin.manager.migrate.process');
    $instance->destinationPluginManager = $container->get('plugin.manager.migrate.destination');
    $instance->migrateLookup = $container->get('migrate.lookup');

    return $instance;
  }

  /**
   * Get the fields for an entity type and bundle.
   */
  public function getSourceFieldInstances(string $entity_type_id, string $bundle = NULL) : ?array {
    $ret = [];

    $query = Database::getConnection('default', 'legacy')->select('field_config', 'fc');
    $query->innerJoin('field_config_instance', 'fci', 'fci.field_id = fc.id');
    $query->addField('fc', 'field_name');
    $query->addField('fc', 'type');
    $query->condition('entity_type', $entity_type_id);
    $query->addField('fci', 'bundle');

    if ($bundle) {
      $query->condition('fci.bundle', $bundle);
    }

    $fields = $query->execute()->fetchAll();

    switch ($entity_type_id) {
      case 'node':
        $types = Database::getConnection('default', 'legacy')->select('node_type', 't')->fields('t', ['type'])->execute()->fetchCol() ?: [];

        foreach ($types as $type) {
          $ret[$type] = [];
        }

        break;

      case 'file':
        $types = Database::getConnection('default', 'legacy')->select('file_type', 't')->fields('t', ['type'])->execute()->fetchCol() ?: [];

        foreach ($types as $type) {
          $ret[$type] = [
            'fid' => 'fid',
            'uri' => 'uri',
          ];
        }

        break;

      case 'taxonomy_term':
        $types = Database::getConnection('default', 'legacy')->select('taxonomy_vocabulary', 't')->fields('t', ['machine_name'])->execute()->fetchCol() ?: [];

        foreach ($types as $type) {
          $ret[$type] = [];
        }

        break;

      case 'user':
        $ret['user'] = [
          'picture' => 'image',
        ];

        break;
    }

    foreach ($fields as &$field) {
      $ret[$field->bundle][$field->field_name] = $field->type;
    }

    if ($entity_type_id === 'file') {
      unset($ret['audio']['fid']);
      unset($ret['audio']['uri']);
      unset($ret['image']['fid']);
      unset($ret['image']['uri']);
      unset($ret['video']['fid']);
      unset($ret['video']['uri']);
    }

    if ($bundle) {
      return $ret[$bundle] ?? NULL;
    }

    return $ret;
  }

  /**
   * Get the fields for an entity type and bundle.
   */
  public function getDestFieldInstances(string $entity_type_id, string $bundle = NULL) : ?array {
    $ret = [];

    $bundle_entity_type = $this->entityTypeManager->getStorage($entity_type_id)->getEntityType()->getBundleEntityType();

    if (!$bundle_entity_type) {
      $bundle_ids = [$entity_type_id];
    }
    else {
      $bundle_ids = [];

      foreach ($this->entityTypeManager->getStorage($bundle_entity_type)->loadMultiple() as $bundle_definition) {
        $bundle_ids[] = $bundle_definition->id();
      }
    }

    foreach ($bundle_ids as $bundle_id) {
      if ($bundle && $bundle_id !== $bundle) {
        continue;
      }

      $ret[$bundle_id] = [];

      $ret[$bundle_id] = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);

      foreach ($ret[$bundle_id] as $field_name => $field_definition) {
        if ($field_definition instanceof BaseFieldDefinition || $field_definition instanceof BaseFieldOverride) {
          unset($ret[$bundle_id][$field_name]);
          continue;
        }

        $ret[$bundle_id][$field_name] = $field_definition->getType();
      }
    }

    if ($bundle) {
      return $ret[$bundle] ?? NULL;
    }

    return $ret;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'monarch_migration_d7_field_map';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['monarch_migration_d7.settings'];
  }

  /**
   * The ajax event.
   */
  public function ajax(array &$form, FormStateInterface $form_state) {
    return $form['map_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $settings = $this->config('monarch_migration_d7.settings')->get('map_settings');

    $form['#tree'] = TRUE;

    $ajax = [
      'callback' => '::ajax',
      'event' => 'change',
      'wrapper' => 'map-settings',
    ];

    $use_map_settings = $form_state->getValue('use_map_settings') ?? $this->config('monarch_migration_d7.settings')->get('use_map_settings') ?? NULL;

    $form['use_map_settings'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use configuration based mapping'),
      '#default_value' => $use_map_settings,
      '#ajax' => $ajax,
    ];

    $form['map_settings']['#prefix'] = '<div id="map-settings">';
    $form['map_settings']['#suffix'] = '</div>';

    if ($use_map_settings) {
      foreach ([
        'user' => 'user',
        'taxonomy_term' => 'taxonomy_term',
        'node' => 'node',
      ] as $source_entity_id => $dest_entity_id) {
        $source_bundles = $this->getSourceFieldInstances($source_entity_id);
        $dest_bundles = $this->getDestFieldInstances($dest_entity_id);

        $form['map_settings'][$source_entity_id]['#type'] = 'fieldset';
        $form['map_settings'][$source_entity_id]['#attributes']['style'] = 'background-color:#ededed';
        $form['map_settings'][$source_entity_id]['#title'] = $source_entity_id . ' => ' . $dest_entity_id;

        foreach ($source_bundles as $source_bundle => $source_fields) {
          $bundle_options = array_combine(array_keys($dest_bundles), array_keys($dest_bundles));

          foreach ($bundle_options as $key => $label) {
            $bundle_options[$key] = $dest_entity_id . '.' . $label;
          }

          $default_value = $settings[$source_entity_id][$source_bundle . '/bundle'] ?? $settings[$source_entity_id][$source_bundle . '/selector'] ?? NULL;

          $form['map_settings'][$source_entity_id][$source_bundle . '/bundle'] = [
            '#type' => 'select',
            '#title' => $this->t('<u>@entity_type.@bundle</u>&nbsp;<i>source bundle</i> maps to:', [
              '@entity_type' => $source_entity_id,
              '@bundle' => $source_bundle,
            ]),
            '#default_value' => $default_value,
            // $this->config('monarch_migration_d7.settings')->get('example'),
            '#empty_option' => $this->t('- Do Not Migrate -'),
            '#empty_value' => '',
            '#options' => $bundle_options,
            '#required' => FALSE,
            '#ajax' => $ajax,
          ];

          $form['map_settings'][$source_entity_id][$source_bundle] = [];

          $dest_bundle = $form_state->getValue([
            'map_settings',
            $source_entity_id,
            $source_bundle . '/bundle',
          ], NULL) ?: $default_value;

          if (!$dest_bundle) {
            $form['map_settings'][$source_entity_id][$source_bundle . '/bundle']['#attributes']['style'] = 'background-color:#ffd6d6';
          }

          if ($dest_bundle && $dest_bundle !== '-') {
            foreach ($source_fields as $source_field_name => $source_field_type) {
              $dest_fields = [];

              foreach (array_keys($dest_bundles[$dest_bundle]) as $key) {
                $dest_fields[$dest_bundles[$dest_bundle][$key]][$key] = $dest_entity_id . '.' . $dest_bundle . '.' . $key;
              }

              $field_value = $form_state->getValue([
                'map_settings',
                $source_entity_id,
                $source_bundle,
                $source_field_name,
              ], NULL) ?: $settings[$source_entity_id][$source_bundle][$source_field_name] ?? NULL;

              $form['map_settings'][$source_entity_id][$source_bundle]['#type'] = 'fieldset';
              $form['map_settings'][$source_entity_id][$source_bundle]['#attributes']['style'] = 'background-color:#fff';
              $form['map_settings'][$source_entity_id][$source_bundle]['#title'] = $this->t('<legend>@source => @dest</legend>', [
                '@source' => $source_bundle,
                '@dest' => $dest_bundle,
              ]);
              $form['map_settings'][$source_entity_id][$source_bundle][$source_field_name] = [
                '#type' => 'select',
                '#title' => $this->t('<u>@entity_type.@bundle.@source</u>&nbsp;<i>"@source_type" field</i> maps to:', [
                  '@source_type' => $source_field_type,
                  '@entity_type' => $source_entity_id,
                  '@bundle' => $source_bundle,
                  '@source' => $source_field_name,
                ]),
                '#default_value' => $settings[$source_entity_id][$source_bundle][$source_field_name] ?? NULL,
                '#empty_option' => $this->t('- Ignore Field -'),
                '#empty_value' => '',
                '#options' => $dest_fields,
                '#required' => FALSE,
                '#ajax' => $ajax,
              ];

              if (!$field_value) {
                $form['map_settings'][$source_entity_id][$source_bundle][$source_field_name]['#attributes']['style'] = 'background-color:#fff5d6';
              }
            }
          }
        }
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('monarch_migration_d7.settings')
      ->set('use_map_settings', $form_state->getValue('use_map_settings'))
      ->set('map_settings', $form_state->getValue('map_settings'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
