<?php

namespace Drupal\monarch_migration_d7\Plugin\MigrationMapper;

use Drupal\migrate\Row;
use Drupal\migrate\MigrateSkipRowException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Maps fields.
 *
 * @MigrationMapper(
 *   id = "monarch_migration_d7_map_using_config",
 *   migration = {
 *     "node",
 *     "taxonomy_term",
 *     "user",
 *   },
 *   weight = -9000,
 * )
 */
class MapUsingConfig extends MigrationMapperD7Base {

  /**
   * The map settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The source bundle information.
   *
   * @var array
   */
  protected $sourceBundles = [];

  /**
   * The destination bundle information.
   *
   * @var array
   */
  protected $destBundles = [];

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->config = $container->get('config.factory')->get('monarch_migration_d7.settings');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {
    if (($migration = $this->configuration['migration'] ?? NULL) && $this->config->get('use_map_settings')) {
      $dest_entity_id = $source_entity_id = $migration->getPluginId();

      if ($source_entity_id === 'user') {
        $source_bundle = 'user';

        if (!($destination_bundle = $this->config->get("map_settings.$source_entity_id.$source_bundle/bundle"))) {
          throw new MigrateSkipRowException();
        }

        $role_map = $this->config->get('role_settings');
        $roles = [];

        foreach ($row->getSourceProperty('roles') ?: [] as $rid) {
          $role_name = $role_map[$rid] ?? NULL;

          if (!is_null($role_name)) {
            $roles[] = $role_name;
          }
        }

        $row->setDestinationProperty('roles', $roles);
      }
      elseif ($source_entity_id === 'taxonomy_term') {
        $source_bundle = $row->getSourceProperty('machine_name');

        if (!($destination_bundle = $this->config->get("map_settings.$source_entity_id.$source_bundle/bundle"))) {
          throw new MigrateSkipRowException();
        }

        $row->setDestinationProperty('vid', $destination_bundle);
      }
      elseif ($source_entity_id === 'node') {
        $source_bundle = $row->getSourceProperty('type');

        if (!($destination_bundle = $this->config->get("map_settings.$source_entity_id.$source_bundle/bundle"))) {
          throw new MigrateSkipRowException();
        }

        $row->setDestinationProperty('type', $destination_bundle);
      }

      $source_instances = static::getSourceFieldInstances($source_entity_id);
      $dest_instances = static::getDestFieldInstances($dest_entity_id);

      foreach (($this->config->get("map_settings.$source_entity_id.$source_bundle") ?: []) as $source_field => $dest_field) {
        if (!$dest_field) {
          continue;
        }

        if (is_null($source_instances[$source_bundle][$source_field] ?? NULL)) {
          throw new \Error("Source field destination does not exist: $source_entity_id.$source_bundle.$source_field");
        }

        if (is_null($dest_instances[$destination_bundle][$dest_field] ?? NULL)) {
          throw new \Error("Destination field destination does not exist: $dest_entity_id.$destination_bundle.$dest_field");
        }

        $map_function = $source_instances[$source_bundle][$source_field] . '_to_' . $dest_instances[$destination_bundle][$dest_field];
        $map_function = 'map' . implode('', array_map(function ($value) {
          return ucfirst(strtolower($value));
        }, explode('_', $map_function)));

        if ($dest_field && $row->hasSourceProperty($source_field) && method_exists($this, $map_function)) {
          $dest_value = static::{$map_function}($row->getSourceProperty($source_field), $source_field, $dest_field, $row);
        }
        else {
          if ($source_instances[$source_bundle][$source_field] === $dest_instances[$destination_bundle][$dest_field]) {
            $dest_value = $row->getSourceProperty($source_field);
          }
          else {
            throw new \Error('No field map function: ' . $map_function);
          }
        }

        if ($row->hasDestinationProperty($dest_field)) {
          $prev_value = $row->getDestinationProperty($dest_field);

          if (!is_array($prev_value)) {
            $prev_value = [$prev_value];
          }

          $row->setDestinationProperty($dest_field, [
            ...array_values($prev_value),
            ...array_values($dest_value),
          ]);
        }
        else {
          $row->setDestinationProperty($dest_field, $dest_value);
        }
      }
    }
  }

  /**
   * Map text fields to string.
   */
  protected function mapTextToString($value) {
    return $value;
  }

  /**
   * Map phone fields to telephone.
   */
  protected function mapPhoneToTelephone($value) {
    return $value;
  }

  /**
   * Map text with summary fields.
   */
  protected function mapTextWithSummaryToTextWithSummary($value) {
    if (is_array($value)) {
      foreach ($value as &$subvalue) {
        if (is_array($subvalue)) {
          $subvalue['format'] = 'full_html';
        }
      }
    }

    return $value;
  }

  /**
   * Map text long fields.
   */
  protected function mapTextLongToTextLong($value) {
    if (is_array($value)) {
      foreach ($value as &$subvalue) {
        if (is_array($subvalue)) {
          $subvalue['format'] = 'full_html';
        }
      }
    }

    return $value;
  }

  /**
   * Map text fields.
   */
  protected function mapTextToText($value) {
    if (is_array($value)) {
      foreach ($value as &$subvalue) {
        if (is_array($subvalue)) {
          $subvalue['format'] = 'full_html';
        }
      }
    }

    return $value;
  }

  /**
   * Map image to entity reference.
   */
  protected function mapImageToEntityReference($value) {
    return $value;
  }

  /**
   * Map taxonomy term reference to entity reference.
   */
  protected function mapTaxonomyTermReferenceToEntityReference($value) {
    return $value;
  }

  /**
   * Map file to entity reference.
   */
  protected function mapFileToEntityReference($value) {
    return $value;
  }

  /**
   * Map file to entity reference.
   */
  protected function mapListTextToListString($value) {
    return $value;
  }

  /**
   * Map phone fields to telephone.
   */
  protected function mapTextToLink($value) {
    if (is_array($value)) {
      foreach ($value as $delta => $subvalue) {
        if ($text = $subvalue['value'] ?? NULL) {
          [
            'scheme' => $scheme,
            'host' => $host,
            'path' => $path,
            'query' => $query,
            'fragment' => $fragment,
          ] = parse_url($text) + [
            'scheme' => 'http',
            'host' => '',
            'path' => '',
            'query' => '',
            'fragment' => '',
          ];

          if ($host) {
            $url = $scheme . '://' . $host . $path;

            if ($query) {
              $url .= '?' . $query;
            }

            if ($fragment) {
              $url .= '#' . $fragment;
            }

            $value[$delta] = [
              'uri' => $url,
              'title' => $url,
            ];
          }
          else {
            $value[$delta] = [
              'uri' => NULL,
              'title' => $text,
            ];
          }
        }
      }
    }

    return $value;
  }

  /**
   * Map text with summary fields.
   */
  protected function mapTaxonomyTermReferenceToListString($value, $field_name, $dest_field_name) {
    if (is_array($value)) {
      foreach ($value as $delta => $subvalue) {
        if (is_array($subvalue) && isset($subvalue['target_id'])) {
          $term = $this->sourceDatabase->select('taxonomy_term_data', 't')->condition('tid', $subvalue['target_id'])->fields('t')->execute()->fetch();
          $storage_definition = $this->entityFieldManager->getFieldStorageDefinitions($this->configuration['migration']->getPluginId())[$dest_field_name] ?? NULL;
          $allowed_values = $storage_definition->getSettings()['allowed_values'] ?? [];

          foreach ($allowed_values as $key => $label) {
            if (strtolower($term->name) === strtolower($label)) {
              $value[$delta] = [
                'value' => $key,
              ];

              continue 2;
            }
          }

          $value[$delta] = NULL;
        }
      }
    }

    return $value;
  }

  /**
   * Map text with summary fields.
   */
  protected function mapTextWithSummaryToEntityReferenceRevisions($value, $field_name, $dest_field_name, Row $row) {
    if (!empty($value)) {
      $ret = [];

      $source_ids = $row->getSourceIdValues();
      asort($source_ids);

      $paragraph_base_key = implode(';', [
        $this->configuration['migration']->getPluginId(),
        serialize($source_ids),
        $field_name,
      ]);

      $ret[] = $this->ref($container = $this->createParagraph("$paragraph_base_key;container", 'mbp_layout', [
        'status' => 1,
      ], [
        'layout_paragraphs' => [
          'region' => '',
          'parent_uuid' => '',
          'layout' => 'monarch_bootstrap_container',
          'config' => [
            'label' => '',
            'container' => 'container',
          ],
        ],
      ]));

      $ret[] = $this->ref($this->createParagraph("$paragraph_base_key;body", 'mp_text', [
        'status' => 1,
        'mp_text' => array_map(function ($values) {
          return ['format' => 'full_html'] + $values;
        }, $value),
      ], [
        'layout_paragraphs' => [
          'region' => 'content',
          'parent_uuid' => $container->uuid(),
          'layout' => '',
          'config' => [
            'label' => '',
          ],
        ],
      ]));

      return $ret;
    }

    return [];
  }

}
