<?php

namespace Drupal\monarch_migration_d7\Plugin\MigrationMapper;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Url;
use Drupal\migrate\Row;
use Drupal\migrate\MigrateSkipRowException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Maps fields.
 *
 * @MigrationMapper(
 *   id = "monarch_migration_d7_menu_link_content",
 *   migration = "menu_link_content",
 *   weight = -9999,
 * )
 */
class MenuLinkContent extends MigrationMapperD7Base {

  const VALIDATE_ROUTE = FALSE;

  /**
   * The path validator service.
   *
   * @var \Drupal\Core\Path\PathValidatorInterface
   */
  protected $pathValidator;

  /**
   * The menu link plugin manager.
   *
   * @var \Drupal\Core\Menu\MenuLinkManagerInterface
   */
  protected $menuLinkManager;

  /**
   * The menu link entity storage handler.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $menuLinkStorage;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->pathValidator = $container->get('path.validator');
    $instance->menuLinkManager = $container->get('plugin.manager.menu.link');
    $instance->menuLinkStorage = $container->get('entity_type.manager')->getStorage('menu_link_content');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function requirements() : ?array {
    return ['menu'];
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {
    $skip_translation = $row->getSourceProperty('skip_translation');

    if (empty($skip_translation) || preg_match('/^(devel|shortcut-set-)/', $row->getSourceProperty('menu_name'))) {
      throw new MigrateSkipRowException();
    }

    $row->setDestinationProperty('skip_translation', $skip_translation);
    $row->setDestinationProperty('langcode', $row->getSourceProperty('language') ?: 'und');
    $row->setDestinationProperty('bundle', 'menu_link_content');
    $row->setDestinationProperty('title', $row->getSourceProperty('link_title'));
    $row->setDestinationProperty('description', $row->getSourceProperty('description'));
    $row->setDestinationProperty('external', $row->getSourceProperty('external'));
    $row->setDestinationProperty('weight', $row->getSourceProperty('weight'));
    $row->setDestinationProperty('expanded', $row->getSourceProperty('external'));
    $row->setDestinationProperty('enabled', $row->getSourceProperty('enabled'));
    $row->setDestinationProperty('changed', $row->getSourceProperty('updated'));
    $row->setDestinationProperty('menu_name', $this->machineName('mig-' . $row->getSourceProperty('menu_name'), '-'));
    $row->setDestinationProperty('link', [
      'uri' => $this->linkUri($row->getSourceProperty('link_path')),
      'options' => $row->getSourceProperty('options'),
    ]);

    $route = $this->route($row->getSourceProperty('link_path'), $row->getSourceProperty('options'));
    $row->setDestinationProperty('route', $route);
    $row->setDestinationProperty('route_name', $route['route_name'] ?? NULL);
    $row->setDestinationProperty('route_parameters', $route['route_parameters'] ?? NULL);
    $row->setDestinationProperty('url', $route['url'] ?? NULL);
    $row->setDestinationProperty('options', $route['options'] ?? NULL);

    $parent = $this->menuLinkParent($row->getSourceProperty('plid'), $row->getDestinationProperty('menu_name'), $row->getSourceProperty('parent_link_path'));
    $row->setDestinationProperty('parent', $parent);
  }

  /**
   * Emulate the link_uri plugin.
   */
  public function linkUri($value) {
    $path = ltrim($value, '/');

    if (parse_url($path, PHP_URL_SCHEME) === NULL) {
      if ($path == '<front>') {
        $path = '';
      }
      elseif ($path == '<nolink>') {
        return 'route:<nolink>';
      }
      $path = 'internal:/' . $path;

      // Convert entity URIs to the entity scheme, if the path matches a route
      // of the form "entity.$entity_type_id.canonical".
      // @see \Drupal\Core\Url::fromEntityUri()
      $url = Url::fromUri($path);
      if ($url->isRouted()) {
        $route_name = $url->getRouteName();
        foreach (array_keys($this->entityTypeManager->getDefinitions()) as $entity_type_id) {
          if ($route_name == "entity.$entity_type_id.canonical" && isset($url->getRouteParameters()[$entity_type_id])) {
            return "entity:$entity_type_id/" . $url->getRouteParameters()[$entity_type_id];
          }
        }
      }
      else {
        // If the URL is not routed, we might want to get something back to do
        // other processing. If this is the case, the "validate_route"
        // configuration option can be set to FALSE to return the URI.
        if (!static::VALIDATE_ROUTE) {
          return $url->getUri();
        }
        else {
          throw new MigrateSkipRowException(sprintf('The path "%s" failed validation.', $path));
        }
      }
    }
    return $path;
  }

  /**
   * Emulate the route plugin.
   */
  public function route(string $link_path, array $options = []) {
    $extracted = $this->pathValidator->getUrlIfValidWithoutAccessCheck($link_path);
    $route = [];

    if ($extracted) {
      if ($extracted->isExternal()) {
        $route['route_name'] = NULL;
        $route['route_parameters'] = [];
        $route['options'] = $options;
        $route['url'] = $extracted->getUri();
      }
      else {
        $route['route_name'] = $extracted->getRouteName();
        $route['route_parameters'] = $extracted->getRouteParameters();
        $route['options'] = $extracted->getOptions();

        if (isset($options['query'])) {
          // If the querystring is stored as a string (as in D6), convert it
          // into an array.
          if (is_string($options['query'])) {
            parse_str($options['query'], $old_query);
          }
          else {
            $old_query = $options['query'];
          }
          $options['query'] = $route['options']['query'] + $old_query;
          unset($route['options']['query']);
        }
        $route['options'] = $route['options'] + $options;
        $route['url'] = NULL;
      }
    }

    return $route;
  }

  /**
   * Emulate the menu_link_parent plugin.
   */
  public function menuLinkParent(?string $parent_id, string $menu_name, ?string $parent_link_path) {
    // Handle root elements of a menu.
    if (!$parent_id) {
      return '';
    }

    $lookup_result = $this->migrateLookup->lookup($this->configuration['migration']->id(), [$parent_id]);
    if ($lookup_result) {
      $already_migrated_id = $lookup_result[0]['id'];
    }

    if (!empty($already_migrated_id) && ($link = $this->menuLinkStorage->load($already_migrated_id))) {
      return $link->getPluginId();
    }

    // Parent could not be determined by ID, so we try to determine by the
    // combination of the menu name and parent link path.
    // If the parent link path is external, URL will be useless because the
    // link will definitely not correspond to a Drupal route.
    if (UrlHelper::isExternal($parent_link_path)) {
      $links = $this->menuLinkStorage->loadByProperties([
        'menu_name' => $menu_name,
        'link.uri' => $parent_link_path,
      ]);
    }
    else {
      $url = Url::fromUserInput('/' . ltrim($parent_link_path, '/'));
      if ($url->isRouted()) {
        $links = $this->menuLinkManager->loadLinksByRoute($url->getRouteName(), $url->getRouteParameters(), $menu_name);
      }
    }
    if (!empty($links)) {
      return reset($links)->getPluginId();
    }

    // Parent could not be determined.
    throw new MigrateSkipRowException(sprintf("No parent link found for plid '%d' in menu '%s'.", $parent_id, $menu_name));
  }

}
