<?php

namespace Drupal\monarch_package_reporter;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Url;
use Drupal\monarch_package_reporter\Form\SettingsForm;
use GuzzleHttp\Client;
use Monarch\ComposerInformation\ComposerInformationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\Routing\RequestContext;

/**
 * Reports package information.
 */
class PackageReporter {

  use ComposerInformationTrait;

  /**
   * Guzzle client.
   */
  protected Client $client;

  /**
   * The request context.
   */
  protected RequestContext $requestContext;

  /**
   * The logger.
   */
  protected LoggerInterface $logger;

  public function __construct(RequestContext $request_context, LoggerChannelFactoryInterface $logger_factory) {
    $this->client = new Client([
      'verify' => FALSE,
    ]);
    $this->requestContext = $request_context;
    $this->logger = $logger_factory->get('monarch_package_reporter');
  }

  /**
   * Check if host name is invalid.
   */
  public function isInvalidHost(string $host) : bool {
    if ($this->requestContext->getHost() === 'default') {
      return TRUE;
    }

    $host_parts = explode('.', $this->requestContext->getHost());
    $host_root = implode('.', array_slice($host_parts, -2, 2));
    $host_suffix = end($host_parts);

    return in_array($host_suffix, ['example', 'invalid', 'localhost', 'local', 'onion', 'test', 'alt']) ||
      in_array($host_root, ['example.com', 'example.net', 'example.org']);
  }

  /**
   * Update remote server with package information.
   */
  public function updateRemote(): bool {
    $settings = \Drupal::config(SettingsForm::CONFIG_NAME);
    $key = $settings->get('key');
    $host = $settings->get('host');

    if (empty($key) || empty($host)) {
      // Feature not enabled.
      return FALSE;
    }

    $driver_type = NULL;
    $server_version = NULL;

    try {
      $conn = \Drupal::database()->getClientConnection();
      $driver_type = \Drupal::database()->databaseType();
      $server_version = $conn->getAttribute(\PDO::ATTR_SERVER_VERSION);
    }
    catch (\Exception $e) {
      // Ignore.
    }

    $data = [
      'info' => [
        'php_version' => phpversion() ?: '',
        'os_type' => php_uname('s') ?: '',
        'os_name' => php_uname('s') ?: '',
        'os_host' => php_uname('n') ?: '',
        'os_release' => php_uname('r') ?: '',
        'os_version' => php_uname('v') ?: '',
        'os_arch' => php_uname('m') ?: '',
        'db_driver' => $driver_type ?: '',
        'db_version' => $server_version ?: '',
        'type' => 'drupal',
        'version' => \Drupal::VERSION,
      ],
      'composer' => $this->getComposerJson(),
      'installed' => $this->getComposerInstalledJson(),
    ];

    if (strtolower($data["info"]["os_name"]) === 'linux') {
      $values = [];

      foreach (preg_split('/[\n\r]+/', @file_get_contents('/etc/os-release') ?: '') as $line) {
        $parts = array_map('trim', explode('=', $line, 2));

        if ($parts[0] && $parts[1]) {
          $values[$parts[0]] = trim($parts[1] ?? '', '"');
        }
      }

      if (isset($values['NAME'])) {
        $data['info']['os_name'] = $values['NAME'];
      }

      if (isset($values['VERSION'])) {
        $data['info']['os_version'] = explode(' ', trim($values['VERSION']), 2)[0];
      }
    }

    try {
      if ($this->isInvalidHost($this->requestContext->getHost())) {
        throw new \Exception('Invalid host. Please ensure that your site configuration is correct. If cron is running from Drush CLI, you may need to specify the URI or create a drush/drush.yml file.');
      }

      $response = $this->client->post("https://$host/update-hub/update", [
        'headers' => [
          'Content-Type' => 'application/json',
          'X-SITE-KEY' => $key,
          'X-SITE-URL' => Url::fromUserInput('/', ['absolute' => TRUE])->toString(),
        ],
        'body' => json_encode($data),
      ]);

      return $response->getStatusCode() === 200;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to send package information: @message', ['@message' => $e->getMessage()]);

      return FALSE;
    }
  }

}
