<?php

namespace Drupal\monarch_package_reporter\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Local settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  const CONFIG_NAME = 'monarch_package_reporter.settings';

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'monarch_package_reporter_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [self::CONFIG_NAME];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Key'),
      '#description' => $this->t('Entering a key here implies that this feature is enabled.'),
      '#default_value' => $this->config(self::CONFIG_NAME)->get('key'),
    ];

    $form['host'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Update Server Host'),
      '#description' => $this->t('Entering a server host here implies that this feature is enabled.'),
      '#placeholder' => 'domain-name.com',
      '#default_value' => $this->config(self::CONFIG_NAME)->get('host') ?: NULL,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $host = trim($form_state->getValue('host') ?: '');

    if ($host && !preg_match('/^([a-z0-9-]+\.)+[a-z0-9-]+$/', $host)) {
      $form_state->setErrorByName(
        'host',
        $this->t('The host domain name is incorrect.'),
      );
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config(self::CONFIG_NAME)
      ->set('key', $form_state->getValue('key'))
      ->set('host', trim($form_state->getValue('host') ?: '') ?: NULL)
      ->save();
    parent::submitForm($form, $form_state);
  }

}
