<?php

namespace Drupal\monarch_package_reporter;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Url;
use Drupal\monarch_package_reporter\Form\SettingsForm;
use Drupal\system\SystemManager;
use Drupal\update\UpdateManagerInterface;
use GuzzleHttp\Client;
use Monarch\ComposerInformation\ComposerInformationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\Routing\RequestContext;

/**
 * Reports package information.
 */
class PackageReporter {

  use ComposerInformationTrait;

  /**
   * Guzzle client.
   */
  protected Client $client;

  /**
   * The request context.
   */
  protected RequestContext $requestContext;

  /**
   * The logger.
   */
  protected LoggerInterface $logger;

  /**
   * The module handler service.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The system manager service.
   */
  protected SystemManager $systemManager;

  public function __construct(RequestContext $request_context, LoggerChannelFactoryInterface $logger_factory, ModuleHandlerInterface $module_handler, SystemManager $system_manager) {
    $this->client = new Client([
      'verify' => FALSE,
    ]);
    $this->requestContext = $request_context;
    $this->logger = $logger_factory->get('monarch_package_reporter');
    $this->moduleHandler = $module_handler;
    $this->systemManager = $system_manager;
  }

  /**
   * Check if host name is invalid.
   */
  public function isInvalidHost(string $host) : bool {
    if ($this->requestContext->getHost() === 'default') {
      return TRUE;
    }

    $host_parts = explode('.', $this->requestContext->getHost());
    $host_root = implode('.', array_slice($host_parts, -2, 2));
    $host_suffix = end($host_parts);

    return in_array($host_suffix, ['example', 'invalid', 'localhost', 'local', 'onion', 'test', 'alt']) ||
      in_array($host_root, ['example.com', 'example.net', 'example.org']);
  }

  /**
   * Update remote server with package information.
   */
  public function updateRemote(): bool {
    $settings = \Drupal::config(SettingsForm::CONFIG_NAME);
    $license_key = $settings->get('key');
    $host = $settings->get('host');

    if (empty($license_key) || empty($host)) {
      // Feature not enabled.
      return FALSE;
    }

    $driver_type = NULL;
    $server_version = NULL;

    try {
      $conn = \Drupal::database()->getClientConnection();
      $driver_type = \Drupal::database()->databaseType();
      $server_version = $conn->getAttribute(\PDO::ATTR_SERVER_VERSION);
    }
    catch (\Exception $e) {
      // Ignore.
    }

    $data = [
      'metadata' => [
        'php_version' => phpversion() ?: '',
        'os_type' => php_uname('s') ?: '',
        'os_name' => php_uname('s') ?: '',
        'os_host' => php_uname('n') ?: '',
        'os_release' => php_uname('r') ?: '',
        'os_version' => php_uname('v') ?: '',
        'os_arch' => php_uname('m') ?: '',
        'db_driver' => $driver_type ?: '',
        'db_version' => $server_version ?: '',
        'type' => 'drupal',
        'version' => \Drupal::VERSION,
        'composer.json' => json_decode($this->getComposerJson(), TRUE),
        'installed.json' => json_decode($this->getComposerInstalledJson(), TRUE),
      ],
      // Core status stuff.
      'status' => [],
    ];

    if (strtolower($data['metadata']['os_name']) === 'linux') {
      $values = [];

      foreach (preg_split('/[\n\r]+/', @file_get_contents('/etc/os-release') ?: '') as $line) {
        $parts = array_map('trim', explode('=', $line, 2));

        if ($parts[0] && $parts[1]) {
          $values[$parts[0]] = trim($parts[1] ?? '', '"');
        }
      }

      if (isset($values['NAME'])) {
        $data['metadata']['os_name'] = $values['NAME'];
      }

      if (isset($values['VERSION'])) {
        $data['metadata']['os_version'] = explode(' ', trim($values['VERSION']), 2)[0];
      }
    }

    $requirements = $this->systemManager->listRequirements();

    foreach ($requirements as $key => $info) {
      $severity = -1;

      if (is_numeric($info['severity'] ?? NULL)) {
        $severity = $info['severity'];
      }
      elseif (is_object($info['severity'] ?? NULL) && method_exists($info['severity'], 'value')) {
        $severity = $info['severity']->value;
      }

      if ($severity != -1 && ($info['title'] ?? NULL) && ($info['value'] ?? NULL)) {
        $data['status']['drupal_requirement'][$key] = [
          'severity' => $severity === 2 ? 'critical' : ($severity === 1 ? 'warning' : 'info'),
          'data' => $info,
        ];
      }
    }

    if (function_exists('update_get_available')) {
      $available = update_get_available(TRUE);
      $this->moduleHandler->loadInclude('update', 'compare.inc');
      $project_data = update_calculate_project_data($available);

      foreach ($project_data as $key => $project) {
        if (($project['status'] ?? -1) === UpdateManagerInterface::CURRENT) {
          continue;
        }

        $severity = [
          UpdateManagerInterface::NOT_SECURE => 'critical',
          UpdateManagerInterface::REVOKED => 'critical',
          UpdateManagerInterface::NOT_SUPPORTED => 'critical',
          UpdateManagerInterface::NOT_CURRENT => 'warning',
          UpdateManagerInterface::CURRENT => 'info',
        ][$project['status'] ?? -1] ?? 'info';

        $data['status']['drupal_project_status'][$key] = [
          'severity' => $severity,
          'data' => $project,
        ];
      }
    }

    try {
      if ($this->isInvalidHost($this->requestContext->getHost())) {
        throw new \Exception('Invalid host. Please ensure that your site configuration is correct. If cron is running from Drush CLI, you may need to specify the URI or create a drush/drush.yml file.');
      }

      $response = $this->client->post("https://$host/update-hub/update", [
        'headers' => [
          'Content-Type' => 'application/json',
          'X-SITE-KEY' => $license_key,
          'X-SITE-URL' => Url::fromUserInput('/', ['absolute' => TRUE])->toString(),
        ],
        'body' => json_encode($data),
      ]);

      return $response->getStatusCode() === 200;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to send package information: @message', ['@message' => $e->getMessage()]);

      return FALSE;
    }
  }

}
