<?php

namespace Drupal\monarch_rest_plugin\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\monarch_rest_plugin\RestApiPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Returns responses for Monarch REST Plugin routes.
 */
final class MonarchRestPluginController extends ControllerBase {

  /**
   * The controller constructor.
   */
  public function __construct(
    private readonly RestApiPluginManager $pluginManagerRestApi,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('plugin.manager.rest_api'),
    );
  }

  /**
   * Call a callback and apply appropriate arguments.
   */
  protected static function callCallback(callable $callback, Request $request = NULL, AccountInterface $account = NULL) {
    $f = new \ReflectionFunction($callback instanceof \Closure ? $callback : \Closure::fromCallable($callback));
    $fp = $f->getParameters();
    $params = [];

    foreach ($fp as $param) {
      if ($param->getName() === 'query' && ((string) $param->getType()) === '?array') {
        $params[$param->getName()] = ($request && $request->query) ? ($request->query->all()) : [];
      }
      elseif (((string) $param->getType()) === '?' . Request::class) {
        $params[$param->getName()] = $request;
      }
      elseif (((string) $param->getType()) === '?' . AccountInterface::class) {
        $params[$param->getName()] = $account;
      }
    }

    if (!empty($params)) {
      return $callback(...$params);
    }

    return $callback();
  }

  /**
   * Builds the response.
   */
  public function build(Request $request = NULL, RouteMatchInterface $route_match = NULL): Response {
    $route = $route_match->getRouteObject()->getPath();
    $callback = $this->pluginManagerRestApi->getRouteCallback($route);

    if ($callback) {
      $ret = static::callCallback($callback, $request, $this->currentUser());

      if ($ret instanceof Response) {
        return $ret;
      }

      return new JsonResponse($ret);
    }

    return new Response(NULL, 404);
  }

  /**
   * Builds the response.
   */
  public function access(Request $request = NULL, RouteMatchInterface $route_match = NULL): AccessResultInterface {
    $route = $route_match->getRouteObject()->getPath();
    $callback = $this->pluginManagerRestApi->getRouteAccessCallback($route);

    if ($callback) {
      $ret = static::callCallback($callback, $request, $this->currentUser());

      if ($ret instanceof AccessResult) {
        return $ret;
      }

      if (is_bool($ret)) {
        return AccessResult::allowedIf((bool) $ret);
      }

      throw new \Exception('Access result must be boolean or implement ' . AccessResult::class);
    }

    return AccessResult::allowedIf($this->currentUser()->hasPermission('access content'));
  }

}
