<?php

namespace Drupal\monarch_rest_plugin\Plugin\RestApi\Rest;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\monarch_rest_plugin\RestApiPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Rest Plugin Discovery API, and also example plugin.
 *
 * The route is taken from the namespace unless specified by the 'route' value
 * in the annotation. In this example, the full class path is:
 *
 *   \Drupal\monarch_rest_plugin\Plugin\RestApi\Rest\Discovery
 *
 * so the route becomes:
 *
 *   /Rest/Discovery
 *
 * automatically. In the case multiple rest API paths collide, then the first
 * plugin found will be used.
 *
 * The 'entry_point' function name is optional. If not specified, then
 * '__invoke' will be used.
 *
 * The 'access' function name is optional. If not specified, then 'access' will
 * be used.
 *
 * @RestApi(
 *   id = "monarch_rest_plugin_discovery",
 *   entry_point = "rest",
 *   access = "access",
 * )
 */
class Discovery extends RestApiPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The plugin manager interface.
   *
   * @var \Drupal\monarch_rest_plugin\RestApiPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);

    $instance->pluginManager = $container->get('plugin.manager.rest_api');

    return $instance;
  }

  /**
   * The default rest API endpoint.
   *
   * The parameters are optional and can be in any order.
   */
  public function rest(array $query = NULL, Request $request = NULL, AccountInterface $account = NULL) {
    $ret = [];

    foreach ($this->pluginManager->getRoutes() as $path => $rest_api_route) {
      /** @var callable $callback */
      $callback = $rest_api_route->getCallback();

      if (is_string($callback)) {
        $ret[$path] = $callback;
      }
      elseif (is_array($callback)) {
        [$class, $function] = $callback + [''];

        if (is_object($class)) {
          $class = get_class($class);
        }

        if ($function) {
          $ret[$path] = "$class::$function";
        }
        else {
          $ret[$path] = "$class";
        }
      }
      elseif ($callback instanceof \Closure) {
        $ret[$path] = 'Closure';
      }
      else {
        $ret[$path] = '?';
      }
    }

    asort($ret);

    return $ret;
  }

  /**
   * The access callback.
   *
   * The parameters are optional and can be in any order. If this function is
   * omitted, then 'access content' is used as the access permission.
   */
  public function access(array $query = NULL, Request $request = NULL, AccountInterface $account = NULL) {
    return $account->hasPermission('access site reports');
  }

}
