<?php

namespace Drupal\monarch_rest_plugin;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\monarch_rest_plugin\Annotation\RestApi;

/**
 * RestApi plugin manager.
 */
final class RestApiPluginManager extends DefaultPluginManager {

  /**
   * The routes.
   */
  protected ?array $routes = NULL;

  /**
   * Constructs the object.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct('Plugin/RestApi', $namespaces, $module_handler, RestApiPluginBase::class, RestApi::class);
    $this->alterInfo('rest_api_info');
    $this->setCacheBackend($cache_backend, 'rest_api_plugins');
  }

  /**
   * Get routes.
   *
   * @return array
   *   The list of routes provided by plugins.
   */
  public function getRoutes() : array {
    if (is_null($this->routes)) {
      $this->routes = [];

      foreach ($this->getDefinitions() as $definition) {
        /**
         * The plugin.
         *
         * @var \Drupal\monarch_rest_plugin\RestApiPluginBase $plugin
         */
        $plugin = $this->createInstance($definition['id']);
        foreach ($plugin->getRoutes() as $route) {
          if ($route instanceof RestApiRoute) {
            $this->routes[$route->getRoute()] = $this->routes[$route->getRoute()] ?? $route;
          }
          else {
            throw new \Exception('The function \'getRoutes\' must return an array of: ' . RestApiRoute::class);
          }
        }
      }
    }

    return $this->routes;
  }

  /**
   * Get an API endpoint callback.
   */
  public function getRoute(string $route) : ?RestApiRoute {
    if ($route[0] !== '/') {
      $route = '/' . $route;
    }

    return $this->getRoutes()[$route] ?? NULL;
  }

  /**
   * Get an API endpoint callback.
   */
  public function getRouteCallback(string $route) : ?callable {
    if ($route[0] !== '/') {
      $route = '/' . $route;
    }

    /** @var \Drupal\monarch_rest_plugin\RestApiRoute | null */
    $route_object = $this->getRoutes()[$route] ?? NULL;

    if (is_null($route_object)) {
      return NULL;
    }

    return $route_object->getCallback();
  }

  /**
   * Overload this class to allow calling endpoints.
   */
  public function __call(string $route, array $arguments) : mixed {
    return $this->getRouteCallback($route)(...$arguments);
  }

  /**
   * Call API endpoint.
   */
  public static function __callStatic(string $route, array $arguments) : mixed {
    return \Drupal::service('plugin.manager.rest_api')->getRouteCallback($route)(...$arguments);
  }

  /**
   * Get an API endpoint callback.
   */
  public function getRouteAccessCallback(string $route) : ?callable {
    if ($route[0] !== '/') {
      $route = '/' . $route;
    }

    /** @var \Drupal\monarch_rest_plugin\RestApiRoute | null */
    $route_object = $this->getRoutes()[$route] ?? NULL;

    if (is_null($route_object)) {
      return NULL;
    }

    return $route_object->getAccessCallback();
  }

}
