<?php

namespace Drupal\monarch_simple_search\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\monarch_simple_search\SimpleSearch;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Returns responses for Monarch Simple Search routes.
 */
final class Search extends ControllerBase {

  /**
   * The simple search service.
   */
  private readonly SimpleSearch $simpleSearch;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $instance = new self();

    $instance->simpleSearch = $container->get('simple_search');
    $instance->moduleHandler = $container->get('module_handler');

    return $instance;
  }

  /**
   * Builds the response.
   */
  public function build(Request $request): Response {
    $start = hrtime(TRUE);

    $query = $request->query->all();
    $q = $query['q'] ?? NULL ?: NULL;
    $f = $query['f'] ?? NULL ?: NULL;
    $f = array_filter(array_values(is_array($f) ? $f : [$f]));
    $page = intval($query['p'] ?? NULL);
    $count = min(100, max(0, intval($query['c'] ?? NULL) ?: 20));

    $this->moduleHandler()->alter('search_query_facets', $f, $request);

    $searchQuery = $this->simpleSearch->query($q, $f);
    $resultCount = is_numeric($query['p'] ?? NULL) ? $searchQuery->countQuery()->execute()->fetchField() : NULL;
    $results = $searchQuery->range($page * $count, $count)
      ->execute()->fetchAll();

    $facets = $this->simpleSearch->facetQuery($q, $f)
      ->execute()->fetchCol();

    $end = hrtime(TRUE);

    $search_response = [
      'results' => $results,
      'facets' => $facets,
    ];

    $this->moduleHandler()->alter('search_response', $search_response);

    $headers = [
      'x-elapsed' => round(($end - $start) / 1000000, 2),
    ];

    if (!is_null($resultCount)) {
      $headers['x-pages'] = ceil($resultCount / $count);
    }

    return new JsonResponse($search_response, 200, $headers);
  }

}
