<?php

namespace Drupal\monarch_simple_search\Form;

use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a monarch_simple_search_index_all form.
 */
final class IndexAll extends FormBase {

  /**
   * The entity_type.manager service.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->entityTypeManager = $container->get('entity_type.manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'monarch_simple_search_index_all';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Index All Content'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $batch = [
      'title' => $this->t('Indexing content...'),
      'operations' => [],
    ];

    $batch['operations'][] = [[$this, 'truncateTables'], []];

    foreach ($this->entityTypeManager->getDefinitions() as $type => $definition) {
      if ($definition instanceof ContentEntityTypeInterface) {
        $storage = $this->entityTypeManager->getStorage($type);
        foreach ($storage->getQuery()->accessCheck(FALSE)->execute() as $id) {
          $batch['operations'][] = [[$this, 'indexEntity'], [$type, $id]];
        }
      }
    }

    batch_set($batch);
  }

  /**
   * Truncate search facets.
   */
  public static function truncateTables(array &$context) {
    $context['message'] = 'Truncating search tables...';
    /** @var \Drupal\monarch_simple_search\SimpleSearch $search */
    $search = \Drupal::service('simple_search');
    $search->truncateTables();
  }

  /**
   * Index an entity.
   */
  public static function indexEntity(string $entity_type_id, string $entity_id, array &$context) {
    $context['message'] = "Indexing $entity_type_id...";
    /** @var \Drupal\monarch_simple_search\SimpleSearch $search */
    $search = \Drupal::service('simple_search');
    $entity = \Drupal::entityTypeManager()->getStorage($entity_type_id)->load($entity_id);
    $search->index($entity);
  }

}
