<?php

namespace Drupal\monarch_simple_search;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\Core\Render\RendererInterface;
use Kaiju\Snowball\StemmerFactory;

/**
 * The simple search service.
 */
final class SimpleSearch {

  /**
   * Stopwords.
   *
   * @var string[]
   */
  protected array $stopwords = [];

  /**
   * Constructs a SimpleSearch object.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly Connection $database,
    private readonly RendererInterface $renderer,
    private readonly ModuleHandlerInterface $moduleHandler,
  ) {
    $stopwords = explode("\n", file_get_contents(__DIR__ . '/../stopwords.txt'));
    $this->stopwords = array_combine($stopwords, $stopwords);
  }

  /**
   * Get keywords from text.
   *
   * @param string $text
   *   The search text.
   * @param string $stemmer_langcode
   *   The language code used for stemming.
   *
   * @return string[]
   *   The keywords.
   */
  public function getKeywords(string $text, ?string $stemmer_langcode = NULL) : array {
    $variants = [];

    $stemmer = $stemmer_langcode ? StemmerFactory::create($stemmer_langcode) : NULL;

    $words = preg_split("/[^a-z0-9\.\-_]+/", strip_tags(strtolower($text)));

    foreach ($words as $word) {
      $word = trim($word, "-_.");

      if ($this->stopwords[$word] ?? NULL) {
        continue;
      }

      $word = $word;
      $variants[$word] = $variants[$word] ?? 0;
      $variants[$word] += 100 * strlen($word);

      if (preg_match("/[^a-z0-9\.]/", $word)) {
        $hyphenated_parts = array_filter(preg_split("/[^a-z0-9\.]/", $word));

        if (count($hyphenated_parts) > 1) {
          foreach ($hyphenated_parts as $hyphenated_part) {
            $variants[$hyphenated_part] = $variants[$hyphenated_part] ?? 0;
            $variants[$hyphenated_part] += 50 * strlen($hyphenated_part);
          }
        }
      }
      elseif ($stemmer) {
        $variant = $stemmer->stem($word);
        $variants[$variant] = $variants[$variant] ?? 0;
        $variants[$variant] += 50 * strlen($variant);
      }
    }

    $keywords = [];

    foreach ($variants as $variant => $weight) {
      if (strlen($variant) > 2 || is_numeric($variant)) {
        $keywords[$variant] = $weight;
      }
    }

    return $keywords;
  }

  /**
   * Index an entity.
   */
  public function index(EntityInterface $entity) : bool {
    $entity_type = $entity->getEntityTypeId();
    $entity_bundle = $entity->bundle();
    $entity_id = $entity->id();
    $entity_language = $entity->language()->getId();

    $this->database->delete('monarch_simple_search_keywords')
      ->condition('type', $entity_type)
      ->condition('bundle', $entity_bundle)
      ->condition('id', $entity_id)
      ->condition('langcode', $entity_language)
      ->execute();
    $this->database->delete('monarch_simple_search_facets')
      ->condition('type', $entity_type)
      ->condition('bundle', $entity_bundle)
      ->condition('id', $entity_id)
      ->condition('langcode', $entity_language)
      ->execute();

    if ($entity instanceof ContentEntityInterface && $entity instanceof EntityPublishedInterface) {
      try {
        $url = $entity->toUrl()->toString();

        if (\str_starts_with($url, '/admin')) {
          return FALSE;
        }
      }
      catch (\Throwable $err) {
        return FALSE;
      }

      if ($entity->isPublished()) {
        $vb = $this->entityTypeManager->getViewBuilder($entity_type);
        $build = $vb->view($entity, 'search_index');

        $search_text = [
          'default' => (string) $this->renderer->renderInIsolation($build),
        ];

        $this->moduleHandler->alter([
          'search_text',
        ], $search_text, $entity);

        $ret = FALSE;

        $keywords = [];

        foreach ($search_text as $entity_subid => $text) {
          if (empty($text) || !is_string($text)) {
            continue;
          }

          $keywords[$entity_subid] = $this->getKeywords($text, $entity_language);

          foreach ($this->getKeywords($entity->label() ?: '', $entity_language) as $key => $value) {
            $keywords[$entity_subid][$key] = max($keywords[$entity_subid][$key] ?? 0, $value * 2);
          }
        }

        $this->moduleHandler->alter('search_keywords', $keywords, $entity);

        foreach ($keywords as $entity_subid => $keyword_list) {
          if (!empty($keyword_list)) {
            $query = $this->database->insert('monarch_simple_search_keywords')->fields([
              'type',
              'bundle',
              'id',
              'subid',
              'id_hash',
              'langcode',
              'keyword',
              'score',
            ]);

            foreach ($keyword_list as $keyword => $weight) {
              $id_hash = hash('sha256', implode(':', [
                $entity_type,
                $entity_bundle,
                $entity_id,
                $entity_subid,
                $entity_language,
              ]));

              $query->values([
                'type' => $entity_type,
                'bundle' => $entity_bundle,
                'id' => $entity_id,
                'subid' => $entity_subid,
                'id_hash' => $id_hash,
                'langcode' => $entity_language,
                'keyword' => $keyword,
                'score' => $weight,
              ]);
            }

            $ret = TRUE;

            try {
              $query->execute();
            }
            catch (\Throwable $err) {
              throw $err;
            }
          }
        }

        $facets = [
          'default' => [
            "entity/$entity_type" => "entity/$entity_type",
            "entity/$entity_type:$entity_bundle" => "entity/$entity_type:$entity_bundle",
          ],
        ];

        foreach ($entity->getFields() as $field_name => $field) {
          if ($field instanceof EntityReferenceFieldItemListInterface && $field->getSetting('target_type') === 'taxonomy_term') {
            foreach ($field as $field_item) {
              /** @var \Drupal\Core\Field\Plugin\Field\FieldType\EntityReferenceItemInterface $field_item */
              /** @var \Drupal\taxonomy\Entity\Term $term */
              if ($term = $field_item->entity ?? NULL) {
                $facets['default'][$term->bundle() . ':' . $term->id()] = $term->bundle() . ':' . $term->id();
              }
            }
          }
        }

        $this->moduleHandler->alter('search_facets', $facets, $entity);

        if (!empty($facets)) {
          $query = $this->database->insert('monarch_simple_search_facets')->fields([
            'type',
            'bundle',
            'id',
            'subid',
            'id_hash',
            'langcode',
            'facet',
          ]);

          foreach ($facets as $entity_subid => $facet_list) {
            foreach ($facet_list as $facet) {
              $id_hash = hash('sha256', implode(':', [
                $entity_type,
                $entity_bundle,
                $entity_id,
                $entity_subid,
                $entity_language,
              ]));

              $query->values([
                'type' => $entity_type,
                'bundle' => $entity_bundle,
                'id' => $entity_id,
                'subid' => $entity_subid,
                'id_hash' => $id_hash,
                'langcode' => $entity_language,
                'facet' => $facet,
              ]);
            }
          }

          try {
            $query->execute();
          }
          catch (\Throwable $err) {
            throw $err;
          }
        }

        return $ret;
      }
    }

    return FALSE;
  }

  /**
   * Index all entities.
   */
  public function indexAll() {
    foreach ($this->entityTypeManager->getDefinitions() as $type => $definition) {
      if ($definition instanceof ContentEntityTypeInterface) {
        $storage = $this->entityTypeManager->getStorage($type);
        foreach ($storage->getQuery()->accessCheck(FALSE)->execute() as $id) {
          if ($entity = $storage->load($id)) {
            $this->index($entity);
          }
        }
      }
    }
  }

  /**
   * Index all entities.
   */
  public function truncateTables() {
    \Drupal::database()->truncate('monarch_simple_search_facets')->execute();
    \Drupal::database()->truncate('monarch_simple_search_keywords')->execute();
  }

  /**
   * Do a simple text search.
   */
  public function query(
    ?string $text = NULL,
    ?array $facets = NULL,
    array $fields = ['type', 'bundle', 'id', 'langcode'],
  ) : SelectInterface {
    $query = $this->database->select('monarch_simple_search_keywords', 's')
      ->fields('s', ['type', 'bundle', 'id', 'subid', 'id_hash', 'langcode']);

    if (!empty($facets)) {
      foreach ($facets as $facet) {
        $facet_query = $this->database->select('monarch_simple_search_facets', 'f')
          ->fields('f', ['id_hash'])
          ->condition('facet', $facet, is_array($facet) ? 'IN' : '=');

        $query->condition('id_hash', $facet_query, 'IN');
      }
    }

    $keywords = array_keys($this->getKeywords($text ?: ''));

    if (!empty($text) && empty($keywords)) {
      $keywords[] = '';
    }

    if (empty($keywords)) {
      $wrapper = $this->database->select($query, 't')->fields('t', $fields)->distinct();

      $wrapper->orderBy('type', 'ASC');
      $wrapper->orderBy('id', 'ASC');
      $wrapper->orderBy('subid', 'ASC');
      $wrapper->orderBy('langcode', 'ASC');

      return $wrapper;
    }

    $query->condition('keyword', $keywords, 'IN');
    $query->addExpression('SUM(score)', 'score');

    $query->groupBy('type');
    $query->groupBy('bundle');
    $query->groupBy('id');
    $query->groupBy('subid');

    $wrapper = $this->database->select($query, 't')->fields('t', $fields);
    $wrapper->addExpression('MAX(score)', 'score');

    $wrapper->groupBy('type');
    $wrapper->groupBy('bundle');
    $wrapper->groupBy('id');

    $wrapper->orderBy('score', 'DESC');
    $wrapper->orderBy('type', 'ASC');
    $wrapper->orderBy('id', 'ASC');
    $wrapper->orderBy('langcode', 'ASC');

    return $this->database->select($wrapper, 't')->fields('t', $fields);
  }

  /**
   * Get facets for a query or array of ids.
   */
  public function facetQuery(?string $text = NULL, ?array $facets = NULL) {
    $facet_query = $this->database->select('monarch_simple_search_facets', 'f')
      ->distinct()
      ->fields('f', ['facet'])
      ->orderBy('facet');

    $facet_query->condition('id_hash', $this->query($text, $facets, ['id_hash']), 'IN');

    return $facet_query;
  }

}
