<?php

namespace Drupal\monarch_vbo\Plugin\Action;

use Drupal\Component\FileSystem\FileSystem;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\Entity\BaseFieldOverride;
use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\field\Entity\FieldConfig;
use Drupal\file\FileInterface;
use Drupal\views_bulk_operations\Action\ViewsBulkOperationsActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use ZipArchive;

/**
 * An action to download a zip of the selected media.
 *
 * @Action(
 *   id = "download_media_action",
 *   label = @Translation("Download"),
 *   type = "",
 *   batch = TRUE,
 * )
 */
class MediaDownloadAction extends ViewsBulkOperationsActionBase implements ContainerFactoryPluginInterface {

  /**
   * The entity_type.manager service.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The file_system service.
   */
  protected FileSystemInterface $fileSystem;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);

    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->fileSystem = $container->get('file_system');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function setContext(array &$context): void {
    parent::setContext($context);

    $this->context['results'] = &$context['results'];
  }

  protected function getArchive(string $id) : ZipArchive {
    $filename = static::getFileName($id);
    $archive = new ZipArchive();
    $archive->open($filename, ZipArchive::CREATE);
    return $archive;
  }

  /**
   * {@inheritdoc}
   */
  public function executeMultiple(array $entities) {
    while (
      empty($this->context['results']['file_id']) ||
      file_exists(static::getFileName($this->context['results']['file_id']))
    ) {
      $this->context['results']['file_id'] = uniqid();
    }

    if ($this->context['redirect_url'] ?? NULL) {
      $this->context['results']['target_url'] = $this->context['redirect_url']->toString();
    }

    $media_groups = [];

    foreach ($entities as $entity) {
      foreach ($entity->getFields() as $field) {
        $field_definition = $field->getFieldDefinition();

        if ($field_definition instanceof BaseFieldDefinition || $field_definition instanceof BaseFieldOverride) {
          continue;
        }

        if ($field instanceof EntityReferenceFieldItemListInterface) {
          $referenced_entities = $field->referencedEntities();

          foreach ($referenced_entities as $referenced_entity) {
            if ($referenced_entity instanceof FileInterface) {
              if (file_exists($referenced_entity->getFileUri())) {
                $media_groups['[' . $entity->id() . ']' . $entity->label()][] = $referenced_entity->getFileUri();
              }
            }
          }
        }
      }
    }

    if ($media_groups) {
      /** @var \Drupal\media\MediaInterface[] $entities */
      $archive = $this->getArchive($this->context['results']['file_id']);

      if ($archive) {
        foreach ($media_groups as $group_name => $file_uris) {
          foreach ($file_uris as $file_uri) {
            $file_uri = $this->fileSystem->realpath($file_uri);
            $new_path = $group_name . '/' . basename($file_uri);
            $archive->addFile($file_uri, $new_path);
          }
        }
  
        $archive->close();
      }
    }

    return array_map(function ($entity) { return 'Added media: ' . $entity->id(); }, $entities);
  }

  /**
   * Executes the plugin.
   */
  public function execute($entity = NULL) {
    throw new \Exception('Unable to execute outside of VBO context.');
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    return $object->access('view', $account, $return_as_object);
  }

  /**
   * Get the file name from the id.
   */
  public static function getFileName(?string $id): string {
    $parts = [
      FileSystem::getOsTemporaryDirectory(),
    ];

    if ($id) {  
      $parts[] = 'media_vbo_download_' . $id . '.zip';
    }

    return implode(DIRECTORY_SEPARATOR, $parts);
  }

  /**
   * {@inheritdoc}
   */
  public static function finished($success, array $results, array $operations): RedirectResponse {
    $url = Url::fromRoute('monarch_vbo.media_vbo_download', [
      'id' => $results['file_id'],
    ])->toString();

    $query = http_build_query([
      'destination' => $results['target_url'],
    ]);

    return new RedirectResponse($url . '?' . $query);
  }

}
