<?php

namespace Drupal\monarch_vbo\Plugin\Action;

use Drupal\Component\FileSystem\FileSystem;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\Entity\BaseFieldOverride;
use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StreamWrapper\LocalStream;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\Core\Url;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use Drupal\views_bulk_operations\Action\ViewsBulkOperationsActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * An action to download a zip of the selected media.
 *
 * @Action(
 *   id = "download_media_action",
 *   label = @Translation("Download"),
 *   type = "media",
 * )
 */
class MediaDownloadAction extends ViewsBulkOperationsActionBase implements ContainerFactoryPluginInterface {

  /**
   * The entity_type.manager service.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The stream wrapper manager.
   */
  protected StreamWrapperManagerInterface $streamWrapperManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);

    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->streamWrapperManager = $container->get('stream_wrapper_manager');

    return $instance;
  }

  /**
   * An enhanced version of realpath that ignores non-local streams.
   */
  public function realpath(string $uri) : string|false {
    $wrapper = $this->streamWrapperManager->getViaUri($uri);

    if ($wrapper) {
      if ($wrapper instanceof LocalStream) {
        return $wrapper->realpath($uri);
      }

      return FALSE;
    }

    return realpath($uri);
  }

  /**
   * {@inheritdoc}
   */
  public function setContext(array &$context): void {
    parent::setContext($context);
  
    $context['sandbox']['files'] ??= [];
    $context['results'] ??= [];
    $context['results']['files'] = &$context['sandbox']['files'];

    $this->context['results'] = &$context['results'];
  }

  /**
   * {@inheritdoc}
   */
  public function executeMultiple(array $entities) {
    $media_files = [];

    foreach ($entities as $entity) {
      foreach ($entity->getFields() as $field) {
        $field_definition = $field->getFieldDefinition();

        if ($field_definition instanceof BaseFieldDefinition || $field_definition instanceof BaseFieldOverride) {
          continue;
        }

        if ($field instanceof EntityReferenceFieldItemListInterface) {
          $referenced_entities = $field->referencedEntities();

          foreach ($referenced_entities as $referenced_entity) {
            if ($referenced_entity instanceof FileInterface) {
              $file_uri = $this->realpath($referenced_entity->getFileUri());

              if ($file_uri && file_exists($file_uri)) {
                $media_files[$file_uri] = $file_uri;
                $this->context['sandbox']['files'][] = $file_uri;
              }
              else {
                throw new \Error('File does not exist or is not locally accessible: ' . $file_uri);
              }
            }
          }
        }
      }
    }

    return $media_files;
  }

  /**
   * Executes the plugin.
   */
  public function execute($entity = NULL) {
    $ret = $this->executeMultiple([$entity]);

    if (is_array($ret)) {
      return reset($ret);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    if (!$account || !$account->hasPermission('download bulk media archive')) {
      return $return_as_object ? AccessResult::forbidden() : FALSE;
    }

    return $object->access('view', $account, $return_as_object);
  }

  /**
   * Get the file name from the id.
   */
  public static function getFileName(?string $id, bool $realpath = TRUE): string {
    $file_system = \Drupal::service('file_system');
    $private_path = $file_system->realpath('private://');
    $public_path = $file_system->realpath('public://');

    $parts = $realpath ? [
      $private_path ?: $public_path,
    ] : [
      $private_path ? 'private://' : 'public://',
    ];

    if ($id) {  
      $parts[] = 'media_vbo_download_' . $id . '.zip';
    }

    return implode(DIRECTORY_SEPARATOR, $parts);
  }

  /**
   * {@inheritdoc}
   */
  public static function finished($success, array $results, array $operations): RedirectResponse {
    assert(!empty($results['files']));

    do {
      $archive_filename = static::getFileName(uniqid() . '_temp', TRUE);
      $archive_uri = static::getFileName(uniqid() . '_temp', FALSE);
    } while (file_exists($archive_filename));

    $archive = new \ZipArchive();
    $archive->open($archive_filename, \ZipArchive::CREATE);

    foreach ($results['files'] as $file_uri) {
      $added = $archive->addFile($file_uri, basename($file_uri), 0, 0, 0);

      if ($added) {
        continue;
      }

      $extension = pathinfo($file_uri, PATHINFO_EXTENSION);
      $filename = pathinfo($file_uri, PATHINFO_FILENAME);

      for ($c = 0; !$added; $c++) {
        $added = $archive->addFile($file_uri, implode('.', [$filename, $c, $extension]), 0, 0, 0);
      }
    }

    $archive->close();

    $file_entity = File::create([
      'langcode' => 'en',
      'uid' => \Drupal::currentUser()->id(),
      'filename' => basename($archive_uri),
      'uri' => $archive_uri,
      'filemime' => 'application/zip',
      'filesize' => filesize($archive_filename),
    ]);

    $file_entity->setTemporary();
    $file_entity->save();

    $hash = hash_file('md5', $archive_filename);
    $download_id = base_convert($hash, 16, 36);
    $new_archive_filename = static::getFileName($download_id);

    /* This is basically what the file system service does, but since we're
     * only working with local files we don't need all the other checks.
     */
    if (!@rename($archive_filename, $new_archive_filename)) {
      if (!@copy($archive_filename, $new_archive_filename)) {
        throw new \Error("Failed to copy $archive_filename to $new_archive_filename");
      }

      @unlink($archive_filename);
    }

    $url = Url::fromRoute('monarch_vbo.media_vbo_download', [
      'id' => $download_id,
    ])->toString();

    return new RedirectResponse($url);
  }

}
